# GAISLER_LICENSE
# ---------------------------------------------------------------------------
# File:        grscrub_crc32_class.py
# Author:      Adria Barros de Oliveira - Frontgrade Gaisler
# Description: Python3 class for computing the CRC32 codes for GRSCRUB usage
#
# DISCLAIMER:
# THIS CODE, AND ALL ACCOMPANYING FILES, DATA AND MATERIALS,
# ARE DISTRIBUTED "AS IS" AND WITH NO WARRANTIES OF ANY KIND,
# WHETHER EXPRESS OR IMPLIED. Good data processing procedure dictates
# that any program be thoroughly tested with non-critical data
# before relying on it. The user must assume the entire risk of
# using the program. THIS DISCLAIMER OF WARRANTY CONSTITUTES AN
# ESSENTIAL PART OF THE AGREEMENT. 
# ---------------------------------------------------------------------------

#############################################################################
# imports
#############################################################################
import sys
from datetime import datetime
import fixedint
import logging
import struct


#############################################################################
# grscrub_crc32 class
#############################################################################

class grscrub_crc32:
    
    ######## constants
    SMAP_WFDR   = 0x30004000
    SMAP_SYNC   = 0xaa995566
    PROCESS_HEADER = True

    ######## public methods

    # initialization
    def __init__(self, debug_level):
        self.__process_state = "INIT"
        self.__num_unsync    = 0
        self.__num_words     = 0
        self.__cram_count    = 0
        self.__cram_words    = 6030690
        self.__total_frames  = 0
        self.__crc32_code    = 0xffffffff
        self.__crc32_frame   = []
        self.__crc32_frame_bin   = []
        self.__debug_level   = debug_level
        if(self.__debug_level > 0):
            try:
                dt = datetime.now()
                logger_name = dt.strftime("%Y%m%d_%H%M%S")
                logger_name = "".join(["./grscrub_crc32_log", "_", logger_name])
                self.__debug_logger = self.__setup_custom_logger(logger_name)
                self.__debug_logger.info("".join(["grscrub_crc32 - ", "logger file created - ", logger_name]))
            except:
                print("ERROR - UNABLE TO INITILIAZE LOGGER - EXITING...")
                return
    
    # set output file name
    def setOutputName(self, output_name):
        self.__output_name = output_name
        self.__debug_print(0, "".join(["grscrub_crc32 - ", "output name - ", self.__output_name]))
    
    # set input files format
    def setInputFormat(self, format):
        self.__format = format
        # check files format
        if format != "txt" and format != "bin" and format != "bit":
            self.__debug_logger.error("".join(["Wrong file format - ", format, " - Allowed formats: txt, bin, or bit."]))
            exit(0)

    # set bit file
    def setBitFile(self, bit_file):
        self.__bit_file_name = bit_file
        self.__debug_print(0, "".join(["grscrub_crc32 - ", "bit file - ", self.__bit_file_name]))
            
    # set mask file
    def setMaskFile(self, mask_file):
        self.__mask_file_name = mask_file
        self.__debug_print(0, "".join(["grscrub_crc32 - ", "mask file - ", self.__mask_file_name]))
            
    # set frame length 
    def setFrameLen(self, frame_len):
        self.__frame_len = frame_len
        self.__debug_print(0, "".join(["grscrub_crc32 - ", "frame length - ", str(self.__frame_len)]))

    # compute CRC32 based on input files
    def processCRC32(self):
        self.__debug_logger.info("".join(["Starting processing files"]))
        
        if self.__format == "txt":
            with open(self.__bit_file_name) as bit_file, open(self.__mask_file_name) as mask_file:
                for bitl, maskl in zip(bit_file, mask_file):
                    # txt files, 1 hex word per line
                    self.__bit_word  = int(bitl.lower().strip(" \n,;"), 16)
                    self.__mask_word = int(maskl.lower().strip(" \n,;"), 16)

                    if self.__process_state != "DONE" and self.PROCESS_HEADER == True:
                        # process data header in bit and mask files
                        self.__process_files()
                    else:
                        # process crc32 per CRAM frame
                        self.__compute_crc32_by_frame()
                        self.__cram_count += 1

                        if self.__cram_count >= self.__cram_words:
                            break

        else:
            with open(self.__bit_file_name, 'rb') as bit_file, open(self.__mask_file_name, 'rb') as mask_file:
                # binary converted to hex
                bitl  = bit_file.read().hex()
                maskl = mask_file.read().hex()
                # remove unsync ASCII header (required since binary may not be aligned)
                i = 0
                while(bitl[i:i+8] != "ffffffff"):
                    i = i+1
                # process
                for shift in range(i, len(bitl), 8):
                    self.__bit_word  = int("".join(["0x",bitl[shift:shift+8]]), 16)
                    self.__mask_word = int("".join(["0x",maskl[shift:shift+8]]), 16)      
                
                    if self.__process_state != "DONE" and self.PROCESS_HEADER == True:
                        # process data header in bit and mask files
                        self.__process_files()
                    else:
                        # process crc32 per CRAM frame
                        self.__compute_crc32_by_frame()
                        self.__cram_count += 1

                        if self.__cram_count >= self.__cram_words:
                            break

        self.__debug_logger.info("".join(["End of CRC32 computation"]))

        if self.__cram_count < self.__cram_words:
            self.__debug_logger.warn("".join(["Bit and mask files are not complete"]))  
            self.__debug_logger.warn("".join(["Total CRAM words read - ", str(self.__cram_count), "; expected number of CRAM words - ", str(self.__cram_words)]))       

        # save output file
        with open("".join([self.__output_name, '.txt']), "w") as output_file_txt, open("".join([self.__output_name, '.bin']), "wb") as output_file_bin: 
            output_file_txt.write('\n'.join(self.__crc32_frame))
            output_file_bin.write(struct.pack("".join(['>', str(len(self.__crc32_frame_bin)), 'I']), *self.__crc32_frame_bin))
            self.__debug_logger.info("".join(["Output files saved"]))


    ######## private functions

    # set logger
    def __setup_custom_logger(self, name):
        formatter = logging.Formatter(fmt='%(asctime)s %(levelname)-8s %(message)s', datefmt='%Y-%m-%d %H:%M:%S')
        handler = logging.FileHandler(''.join([name, '.txt']), mode='w')
        handler.setFormatter(formatter)
        screen_handler = logging.StreamHandler(stream=sys.stdout)
        screen_handler.setFormatter(formatter)
        logger = logging.getLogger(name)
        logger.setLevel(logging.DEBUG)
        logger.addHandler(handler)
        logger.addHandler(screen_handler)
        return logger
    
    # debug print
    def __debug_print(self, level, string):
        if(self.__debug_level > level):
            self.__debug_logger.info(string)

    def __process_files(self):
        if self.__bit_word != self.__mask_word:
            self.__num_unsync += 1
            if self.__num_unsync >= 3:
                self.__debug_logger.error("".join(["Bit and mask files are not synchronized - ", str(self.__bit_word), " ", str(self.__mask_word)]))
                exit(0)

        if self.__process_state == "INIT":
            if self.__bit_word == self.SMAP_SYNC:
                self.__process_state = "FIND_WFDR"
                # self.__debug_logger.info("".join(["SMAP_SYNC found - ", hex(self.__bit_word)]))
        elif self.__process_state == "FIND_WFDR":
            if self.__bit_word == self.SMAP_WFDR:
                self.__process_state = "FIND_SIZE"
                # self.__debug_logger.info("".join(["SMAP_WFDR found - ", hex(self.__bit_word)]))
        elif self.__process_state == "FIND_SIZE":
            self.__cram_words = self.__bit_word & 0x07FFFFFF
            self.__process_state = "DONE"
            self.__debug_logger.info("".join(["Total CRAM words - ", str(self.__cram_words)]))
            self.__debug_logger.info("".join(["Starting CRC32 computation"]))
        else:
            self.__debug_logger.error("".join(["Wrong state - ", self.__process_state]))
            exit(0)
        
    def __compute_crc32_by_frame(self):
        data = self.__bit_word & (~self.__mask_word)
        self.__crc32_code = self.__compute_crc32(data, self.__crc32_code)

        self.__num_words += 1
        if self.__num_words == self.__frame_len:
            self.__crc32_frame.append(f"{self.__crc32_code:#0{10}x}")
            self.__crc32_frame_bin.append(self.__crc32_code)
            if(self.__debug_level > 1):
                self.__debug_logger.info("".join(["Frame id - ", str(self.__total_frames), "; crc32 code - ", f"{self.__crc32_code:#0{10}x}"]))
            self.__crc32_code    = 0xffffffff
            self.__num_words     = 0
            self.__total_frames += 1
        
    def __compute_crc32(self, data_in, crc_init):    
        crc32in = fixedint.Int32(crc_init)
        idata   = fixedint.Int32(data_in)

        odata = ((crc32in[0]  ^ crc32in[3]  ^ crc32in[6]  ^ crc32in[9]  ^ crc32in[12] ^ crc32in[14] ^ crc32in[15] ^ crc32in[20] ^ crc32in[21] ^ crc32in[26] ^ crc32in[27] ^ crc32in[28] ^ crc32in[29] ^ crc32in[31] ^ idata[0] ^ idata[3] ^ idata[6] ^ idata[9] ^ idata[12] ^ idata[14] ^ idata[15] ^ idata[20] ^ idata[21] ^ idata[26] ^ idata[27] ^ idata[28] ^ idata[29] ^ idata[31]) << 0) | \
                ((crc32in[1]  ^ crc32in[4]  ^ crc32in[7]  ^ crc32in[10] ^ crc32in[13] ^ crc32in[15] ^ crc32in[16] ^ crc32in[21] ^ crc32in[22] ^ crc32in[27] ^ crc32in[28] ^ crc32in[29] ^ crc32in[30] ^ idata[1] ^ idata[4] ^ idata[7] ^ idata[10] ^ idata[13] ^ idata[15] ^ idata[16] ^ idata[21] ^ idata[22] ^ idata[27] ^ idata[28] ^ idata[29] ^ idata[30]) << 1) | \
                ((crc32in[2]  ^ crc32in[5]  ^ crc32in[8]  ^ crc32in[11] ^ crc32in[14] ^ crc32in[16] ^ crc32in[17] ^ crc32in[22] ^ crc32in[23] ^ crc32in[28] ^ crc32in[29] ^ crc32in[30] ^ crc32in[31] ^ idata[2] ^ idata[5] ^ idata[8] ^ idata[11] ^ idata[14] ^ idata[16] ^ idata[17] ^ idata[22] ^ idata[23] ^ idata[28] ^ idata[29] ^ idata[30] ^ idata[31]) << 2) | \
                ((crc32in[0]  ^ crc32in[14] ^ crc32in[17] ^ crc32in[18] ^ crc32in[20] ^ crc32in[21] ^ crc32in[23] ^ crc32in[24] ^ crc32in[26] ^ crc32in[27] ^ crc32in[28] ^ crc32in[30] ^ idata[0]    ^ idata[14] ^ idata[17] ^ idata[18] ^ idata[20] ^ idata[21] ^ idata[23] ^ idata[24] ^ idata[26] ^ idata[27] ^ idata[28] ^ idata[30]) << 3) | \
                ((crc32in[1]  ^ crc32in[15] ^ crc32in[18] ^ crc32in[19] ^ crc32in[21] ^ crc32in[22] ^ crc32in[24] ^ crc32in[25] ^ crc32in[27] ^ crc32in[28] ^ crc32in[29] ^ crc32in[31] ^ idata[1]    ^ idata[15] ^ idata[18] ^ idata[19] ^ idata[21] ^ idata[22] ^ idata[24] ^ idata[25] ^ idata[27] ^ idata[28] ^ idata[29] ^ idata[31]) << 4) | \
                ((crc32in[2]  ^ crc32in[16] ^ crc32in[19] ^ crc32in[20] ^ crc32in[22] ^ crc32in[23] ^ crc32in[25] ^ crc32in[26] ^ crc32in[28] ^ crc32in[29] ^ crc32in[30] ^ idata[2]    ^ idata[16]   ^ idata[19] ^ idata[20] ^ idata[22] ^ idata[23] ^ idata[25] ^ idata[26] ^ idata[28] ^ idata[29] ^ idata[30]) << 5) | \
                ((crc32in[3]  ^ crc32in[17] ^ crc32in[20] ^ crc32in[21] ^ crc32in[23] ^ crc32in[24] ^ crc32in[26] ^ crc32in[27] ^ crc32in[29] ^ crc32in[30] ^ crc32in[31] ^ idata[3]    ^ idata[17]   ^ idata[20] ^ idata[21] ^ idata[23] ^ idata[24] ^ idata[26] ^ idata[27] ^ idata[29] ^ idata[30] ^ idata[31]) << 6) | \
                ((crc32in[4]  ^ crc32in[18] ^ crc32in[21] ^ crc32in[22] ^ crc32in[24] ^ crc32in[25] ^ crc32in[27] ^ crc32in[28] ^ crc32in[30] ^ crc32in[31] ^ idata[4]    ^ idata[18]   ^ idata[21]   ^ idata[22] ^ idata[24] ^ idata[25] ^ idata[27] ^ idata[28] ^ idata[30] ^ idata[31]) << 7) | \
                ((crc32in[5]  ^ crc32in[19] ^ crc32in[22] ^ crc32in[23] ^ crc32in[25] ^ crc32in[26] ^ crc32in[28] ^ crc32in[29] ^ crc32in[31] ^ idata[5]    ^ idata[19]   ^ idata[22]   ^ idata[23]   ^ idata[25] ^ idata[26] ^ idata[28] ^ idata[29] ^ idata[31]) << 8) | \
                ((crc32in[6]  ^ crc32in[20] ^ crc32in[23] ^ crc32in[24] ^ crc32in[26] ^ crc32in[27] ^ crc32in[29] ^ crc32in[30] ^ idata[6]    ^ idata[20]   ^ idata[23]   ^ idata[24]   ^ idata[26]   ^ idata[27] ^ idata[29] ^ idata[30]) << 9) | \
                ((crc32in[7]  ^ crc32in[21] ^ crc32in[24] ^ crc32in[25] ^ crc32in[27] ^ crc32in[28] ^ crc32in[30] ^ crc32in[31] ^ idata[7]    ^ idata[21]   ^ idata[24]   ^ idata[25]   ^ idata[27]   ^ idata[28] ^ idata[30] ^ idata[31]) << 10) | \
                ((crc32in[8]  ^ crc32in[22] ^ crc32in[25] ^ crc32in[26] ^ crc32in[28] ^ crc32in[29] ^ crc32in[31] ^ idata[8]    ^ idata[22]   ^ idata[25]   ^ idata[26]   ^ idata[28]   ^ idata[29]   ^ idata[31]) << 11) | \
                ((crc32in[9]  ^ crc32in[23] ^ crc32in[26] ^ crc32in[27] ^ crc32in[29] ^ crc32in[30] ^ idata[9]    ^ idata[23]   ^ idata[26]   ^ idata[27]   ^ idata[29]   ^ idata[30]) << 12) | \
                ((crc32in[10] ^ crc32in[24] ^ crc32in[27] ^ crc32in[28] ^ crc32in[30] ^ crc32in[31] ^ idata[10]   ^ idata[24]   ^ idata[27]   ^ idata[28]   ^ idata[30]   ^ idata[31]) << 13) | \
                ((crc32in[0]  ^ crc32in[3]  ^ crc32in[6]  ^ crc32in[9]  ^ crc32in[11] ^ crc32in[12] ^ crc32in[14] ^ crc32in[15] ^ crc32in[20] ^ crc32in[21] ^ crc32in[25] ^ crc32in[26] ^ crc32in[27] ^ idata[0] ^ idata[3] ^ idata[6] ^ idata[9] ^ idata[11] ^ idata[12] ^ idata[14] ^ idata[15] ^ idata[20] ^ idata[21] ^ idata[25] ^ idata[26] ^ idata[27]) << 14) | \
                ((crc32in[1]  ^ crc32in[4]  ^ crc32in[7]  ^ crc32in[10] ^ crc32in[12] ^ crc32in[13] ^ crc32in[15] ^ crc32in[16] ^ crc32in[21] ^ crc32in[22] ^ crc32in[26] ^ crc32in[27] ^ crc32in[28] ^ idata[1] ^ idata[4] ^ idata[7] ^ idata[10] ^ idata[12] ^ idata[13] ^ idata[15] ^ idata[16] ^ idata[21] ^ idata[22] ^ idata[26] ^ idata[27] ^ idata[28]) << 15) | \
                ((crc32in[2]  ^ crc32in[5]  ^ crc32in[8]  ^ crc32in[11] ^ crc32in[13] ^ crc32in[14] ^ crc32in[16] ^ crc32in[17] ^ crc32in[22] ^ crc32in[23] ^ crc32in[27] ^ crc32in[28] ^ crc32in[29] ^ idata[2] ^ idata[5] ^ idata[8] ^ idata[11] ^ idata[13] ^ idata[14] ^ idata[16] ^ idata[17] ^ idata[22] ^ idata[23] ^ idata[27] ^ idata[28] ^ idata[29]) << 16) | \
                ((crc32in[3]  ^ crc32in[6]  ^ crc32in[9]  ^ crc32in[12] ^ crc32in[14] ^ crc32in[15] ^ crc32in[17] ^ crc32in[18] ^ crc32in[23] ^ crc32in[24] ^ crc32in[28] ^ crc32in[29] ^ crc32in[30] ^ idata[3] ^ idata[6] ^ idata[9] ^ idata[12] ^ idata[14] ^ idata[15] ^ idata[17] ^ idata[18] ^ idata[23] ^ idata[24] ^ idata[28] ^ idata[29] ^ idata[30]) << 17) | \
                ((crc32in[0]  ^ crc32in[3]  ^ crc32in[4]  ^ crc32in[6]  ^ crc32in[7]  ^ crc32in[9]  ^ crc32in[10] ^ crc32in[12] ^ crc32in[13] ^ crc32in[14] ^ crc32in[16] ^ crc32in[18] ^ crc32in[19] ^ crc32in[20] ^ crc32in[21] ^ crc32in[24] ^ crc32in[25] ^ crc32in[26] ^ crc32in[27] ^ crc32in[28] ^ crc32in[30] ^ idata[0] ^ idata[3] ^ idata[4] ^ idata[6] ^ idata[7] ^ idata[9] ^ idata[10] ^ idata[12] ^ idata[13] ^ idata[14] ^ idata[16] ^ idata[18] ^ idata[19] ^ idata[20] ^ idata[21] ^ idata[24] ^ idata[25] ^ idata[26] ^ idata[27] ^ idata[28] ^ idata[30]) << 18) | \
                ((crc32in[1]  ^ crc32in[4]  ^ crc32in[5]  ^ crc32in[7]  ^ crc32in[8]  ^ crc32in[10] ^ crc32in[11] ^ crc32in[13] ^ crc32in[14] ^ crc32in[15] ^ crc32in[17] ^ crc32in[19] ^ crc32in[20] ^ crc32in[21] ^ crc32in[22] ^ crc32in[25] ^ crc32in[26] ^ crc32in[27] ^ crc32in[28] ^ crc32in[29] ^ crc32in[31] ^ idata[1] ^ idata[4] ^ idata[5] ^ idata[7] ^ idata[8] ^ idata[10] ^ idata[11] ^ idata[13] ^ idata[14] ^ idata[15] ^ idata[17] ^ idata[19] ^ idata[20] ^ idata[21] ^ idata[22] ^ idata[25] ^ idata[26] ^ idata[27] ^ idata[28] ^ idata[29] ^ idata[31]) << 19) | \
                ((crc32in[2]  ^ crc32in[5]  ^ crc32in[6]  ^ crc32in[8]  ^ crc32in[9]  ^ crc32in[11] ^ crc32in[12] ^ crc32in[14] ^ crc32in[15] ^ crc32in[16] ^ crc32in[18] ^ crc32in[20] ^ crc32in[21] ^ crc32in[22] ^ crc32in[23] ^ crc32in[26] ^ crc32in[27] ^ crc32in[28] ^ crc32in[29] ^ crc32in[30] ^ idata[2] ^ idata[5] ^ idata[6] ^ idata[8] ^ idata[9] ^ idata[11] ^ idata[12] ^ idata[14] ^ idata[15] ^ idata[16] ^ idata[18] ^ idata[20] ^ idata[21] ^ idata[22] ^ idata[23] ^ idata[26] ^ idata[27] ^ idata[28] ^ idata[29] ^ idata[30]) << 20) | \
                ((crc32in[3]  ^ crc32in[6]  ^ crc32in[7]  ^ crc32in[9]  ^ crc32in[10] ^ crc32in[12] ^ crc32in[13] ^ crc32in[15] ^ crc32in[16] ^ crc32in[17] ^ crc32in[19] ^ crc32in[21] ^ crc32in[22] ^ crc32in[23] ^ crc32in[24] ^ crc32in[27] ^ crc32in[28] ^ crc32in[29] ^ crc32in[30] ^ crc32in[31] ^ idata[3] ^ idata[6] ^ idata[7] ^ idata[9] ^ idata[10] ^ idata[12] ^ idata[13] ^ idata[15] ^ idata[16] ^ idata[17] ^ idata[19] ^ idata[21] ^ idata[22] ^ idata[23] ^ idata[24] ^ idata[27] ^ idata[28] ^ idata[29] ^ idata[30] ^ idata[31]) << 21) | \
                ((crc32in[4]  ^ crc32in[7]  ^ crc32in[8]  ^ crc32in[10] ^ crc32in[11] ^ crc32in[13] ^ crc32in[14] ^ crc32in[16] ^ crc32in[17] ^ crc32in[18] ^ crc32in[20] ^ crc32in[22] ^ crc32in[23] ^ crc32in[24] ^ crc32in[25] ^ crc32in[28] ^ crc32in[29] ^ crc32in[30] ^ crc32in[31] ^ idata[4] ^ idata[7] ^ idata[8] ^ idata[10] ^ idata[11] ^ idata[13] ^ idata[14] ^ idata[16] ^ idata[17] ^ idata[18] ^ idata[20] ^ idata[22] ^ idata[23] ^ idata[24] ^ idata[25] ^ idata[28] ^ idata[29] ^ idata[30] ^ idata[31]) << 22) | \
                ((crc32in[5]  ^ crc32in[8]  ^ crc32in[9]  ^ crc32in[11] ^ crc32in[12] ^ crc32in[14] ^ crc32in[15] ^ crc32in[17] ^ crc32in[18] ^ crc32in[19] ^ crc32in[21] ^ crc32in[23] ^ crc32in[24] ^ crc32in[25] ^ crc32in[26] ^ crc32in[29] ^ crc32in[30] ^ crc32in[31] ^ idata[5] ^ idata[8] ^ idata[9] ^ idata[11] ^ idata[12] ^ idata[14] ^ idata[15] ^ idata[17] ^ idata[18] ^ idata[19] ^ idata[21] ^ idata[23] ^ idata[24] ^ idata[25] ^ idata[26] ^ idata[29] ^ idata[30] ^ idata[31]) << 23) | \
                ((crc32in[6]  ^ crc32in[9]  ^ crc32in[10] ^ crc32in[12] ^ crc32in[13] ^ crc32in[15] ^ crc32in[16] ^ crc32in[18] ^ crc32in[19] ^ crc32in[20] ^ crc32in[22] ^ crc32in[24] ^ crc32in[25] ^ crc32in[26] ^ crc32in[27] ^ crc32in[30] ^ crc32in[31] ^ idata[6] ^ idata[9] ^ idata[10] ^ idata[12] ^ idata[13] ^ idata[15] ^ idata[16] ^ idata[18] ^ idata[19] ^ idata[20] ^ idata[22] ^ idata[24] ^ idata[25] ^ idata[26] ^ idata[27] ^ idata[30] ^ idata[31]) << 24) | \
                ((crc32in[7]  ^ crc32in[10] ^ crc32in[11] ^ crc32in[13] ^ crc32in[14] ^ crc32in[16] ^ crc32in[17] ^ crc32in[19] ^ crc32in[20] ^ crc32in[21] ^ crc32in[23] ^ crc32in[25] ^ crc32in[26] ^ crc32in[27] ^ crc32in[28] ^ crc32in[31] ^ idata[7] ^ idata[10] ^ idata[11] ^ idata[13] ^ idata[14] ^ idata[16] ^ idata[17] ^ idata[19] ^ idata[20] ^ idata[21] ^ idata[23] ^ idata[25] ^ idata[26] ^ idata[27] ^ idata[28] ^ idata[31]) << 25) | \
                ((crc32in[8]  ^ crc32in[11] ^ crc32in[12] ^ crc32in[14] ^ crc32in[15] ^ crc32in[17] ^ crc32in[18] ^ crc32in[20] ^ crc32in[21] ^ crc32in[22] ^ crc32in[24] ^ crc32in[26] ^ crc32in[27] ^ crc32in[28] ^ crc32in[29] ^ idata[8] ^ idata[11] ^ idata[12] ^ idata[14] ^ idata[15] ^ idata[17] ^ idata[18] ^ idata[20] ^ idata[21] ^ idata[22] ^ idata[24] ^ idata[26] ^ idata[27] ^ idata[28] ^ idata[29]) << 26) | \
                ((crc32in[9]  ^ crc32in[12] ^ crc32in[13] ^ crc32in[15] ^ crc32in[16] ^ crc32in[18] ^ crc32in[19] ^ crc32in[21] ^ crc32in[22] ^ crc32in[23] ^ crc32in[25] ^ crc32in[27] ^ crc32in[28] ^ crc32in[29] ^ crc32in[30] ^ idata[9] ^ idata[12] ^ idata[13] ^ idata[15] ^ idata[16] ^ idata[18] ^ idata[19] ^ idata[21] ^ idata[22] ^ idata[23] ^ idata[25] ^ idata[27] ^ idata[28] ^ idata[29] ^ idata[30]) << 27) | \
                ((crc32in[10] ^ crc32in[13] ^ crc32in[14] ^ crc32in[16] ^ crc32in[17] ^ crc32in[19] ^ crc32in[20] ^ crc32in[22] ^ crc32in[23] ^ crc32in[24] ^ crc32in[26] ^ crc32in[28] ^ crc32in[29] ^ crc32in[30] ^ crc32in[31] ^ idata[10] ^ idata[13] ^ idata[14] ^ idata[16] ^ idata[17] ^ idata[19] ^ idata[20] ^ idata[22] ^ idata[23] ^ idata[24] ^ idata[26] ^ idata[28] ^ idata[29] ^ idata[30] ^ idata[31]) << 28) | \
                ((crc32in[0]  ^ crc32in[3]  ^ crc32in[6]  ^ crc32in[9]  ^ crc32in[11] ^ crc32in[12] ^ crc32in[17] ^ crc32in[18] ^ crc32in[23] ^ crc32in[24] ^ crc32in[25] ^ crc32in[26] ^ crc32in[28] ^ crc32in[30] ^ idata[0] ^ idata[3] ^ idata[6] ^ idata[9] ^ idata[11] ^ idata[12] ^ idata[17] ^ idata[18] ^ idata[23] ^ idata[24] ^ idata[25] ^ idata[26] ^ idata[28] ^ idata[30]) << 29) | \
                ((crc32in[1]  ^ crc32in[4]  ^ crc32in[7]  ^ crc32in[10] ^ crc32in[12] ^ crc32in[13] ^ crc32in[18] ^ crc32in[19] ^ crc32in[24] ^ crc32in[25] ^ crc32in[26] ^ crc32in[27] ^ crc32in[29] ^ crc32in[31] ^ idata[1] ^ idata[4] ^ idata[7] ^ idata[10] ^ idata[12] ^ idata[13] ^ idata[18] ^ idata[19] ^ idata[24] ^ idata[25] ^ idata[26] ^ idata[27] ^ idata[29] ^ idata[31]) << 30) | \
                ((crc32in[2]  ^ crc32in[5]  ^ crc32in[8]  ^ crc32in[11] ^ crc32in[13] ^ crc32in[14] ^ crc32in[19] ^ crc32in[20] ^ crc32in[25] ^ crc32in[26] ^ crc32in[27] ^ crc32in[28] ^ crc32in[30] ^ idata[2] ^ idata[5] ^ idata[8] ^ idata[11] ^ idata[13] ^ idata[14] ^ idata[19] ^ idata[20] ^ idata[25] ^ idata[26] ^ idata[27] ^ idata[28] ^ idata[30]) << 31)
        return odata
