/******************** obj2array ****************************
 *
 * Print a binary elf file in  suitable C arrays to stdout.
 * Three arrays are generated: Address, size and data.
 * Allocated but not loaded sections (.bss) are represented
 * by a NULL data element. Section data is appended with
 * a RMAP crc.
 *
 * -e prints the entry point
 * -t prints the highest allocated address + 1
 *
 ***********************************************************/
#include <bfd.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <limits.h>
#include <string.h>

static unsigned char RMAP_CRCTable[256] = {
    0x00, 0x91, 0xe3, 0x72, 0x07, 0x96, 0xe4, 0x75,
    0x0e, 0x9f, 0xed, 0x7c, 0x09, 0x98, 0xea, 0x7b,
    0x1c, 0x8d, 0xff, 0x6e, 0x1b, 0x8a, 0xf8, 0x69,
    0x12, 0x83, 0xf1, 0x60, 0x15, 0x84, 0xf6, 0x67,
    0x38, 0xa9, 0xdb, 0x4a, 0x3f, 0xae, 0xdc, 0x4d,
    0x36, 0xa7, 0xd5, 0x44, 0x31, 0xa0, 0xd2, 0x43,
    0x24, 0xb5, 0xc7, 0x56, 0x23, 0xb2, 0xc0, 0x51,
    0x2a, 0xbb, 0xc9, 0x58, 0x2d, 0xbc, 0xce, 0x5f,
    0x70, 0xe1, 0x93, 0x02, 0x77, 0xe6, 0x94, 0x05,
    0x7e, 0xef, 0x9d, 0x0c, 0x79, 0xe8, 0x9a, 0x0b,
    0x6c, 0xfd, 0x8f, 0x1e, 0x6b, 0xfa, 0x88, 0x19,
    0x62, 0xf3, 0x81, 0x10, 0x65, 0xf4, 0x86, 0x17,
    0x48, 0xd9, 0xab, 0x3a, 0x4f, 0xde, 0xac, 0x3d,
    0x46, 0xd7, 0xa5, 0x34, 0x41, 0xd0, 0xa2, 0x33,
    0x54, 0xc5, 0xb7, 0x26, 0x53, 0xc2, 0xb0, 0x21,
    0x5a, 0xcb, 0xb9, 0x28, 0x5d, 0xcc, 0xbe, 0x2f,
    0xe0, 0x71, 0x03, 0x92, 0xe7, 0x76, 0x04, 0x95,
    0xee, 0x7f, 0x0d, 0x9c, 0xe9, 0x78, 0x0a, 0x9b,
    0xfc, 0x6d, 0x1f, 0x8e, 0xfb, 0x6a, 0x18, 0x89,
    0xf2, 0x63, 0x11, 0x80, 0xf5, 0x64, 0x16, 0x87,
    0xd8, 0x49, 0x3b, 0xaa, 0xdf, 0x4e, 0x3c, 0xad,
    0xd6, 0x47, 0x35, 0xa4, 0xd1, 0x40, 0x32, 0xa3,
    0xc4, 0x55, 0x27, 0xb6, 0xc3, 0x52, 0x20, 0xb1,
    0xca, 0x5b, 0x29, 0xb8, 0xcd, 0x5c, 0x2e, 0xbf,
    0x90, 0x01, 0x73, 0xe2, 0x97, 0x06, 0x74, 0xe5,
    0x9e, 0x0f, 0x7d, 0xec, 0x99, 0x08, 0x7a, 0xeb,
    0x8c, 0x1d, 0x6f, 0xfe, 0x8b, 0x1a, 0x68, 0xf9,
    0x82, 0x13, 0x61, 0xf0, 0x85, 0x14, 0x66, 0xf7,
    0xa8, 0x39, 0x4b, 0xda, 0xaf, 0x3e, 0x4c, 0xdd,
    0xa6, 0x37, 0x45, 0xd4, 0xa1, 0x30, 0x42, 0xd3,
    0xb4, 0x25, 0x57, 0xc6, 0xb3, 0x22, 0x50, 0xc1,
    0xba, 0x2b, 0x59, 0xc8, 0xbd, 0x2c, 0x5e, 0xcf,
};


unsigned char rmap_crc_calc(unsigned char *data, unsigned int len)
{
    unsigned char crc = 0;
    unsigned int i;
    for (i = 0; i < len; i++) {
	crc = RMAP_CRCTable[(crc ^ data[i]) & 0xff];
    }
    return crc;
}

int main(int argc, char** argv) {

    bfd *abfd;
    asection *p;
    char *filename = argv[1];
    int nsec, cursec = 0;
    unsigned int* sectionaddr;
    unsigned int* sectionsize;
    unsigned int** sectiondata;
    unsigned int i, n, x = 1;
    unsigned int startAddress, appEnd = 0;
    int littleend;

    littleend = (int) (((char *)&x)[0]);

    if (argc < 2) {
	fprintf (stderr, "usage: %s elffile [-e|-t]\n", argv[0]);
	fprintf (stderr, "       print allocated sections of elf file as C-arrays to standard out\n");
	fprintf (stderr, "       -e  print entry point\n");
	fprintf (stderr, "       -t  print last allocated RAM address\n");
	fprintf (stderr, "       -t8 print first 8byte boundary above last allocated address\n");
	fprintf (stderr, "       -t4 print first 4kbyte boundary above last allocated address\n");
	fprintf (stderr, "       -t8a <top of ram>  print size after t8 to top of RAM\n");
	fprintf (stderr, "       -t4b <start of ram>  print size before t4 to start of RAM\n");
	return -1;
    }
    if ((abfd = bfd_openr(filename, NULL)) == NULL) {
	fprintf (stderr, "Error opening file '%s': %d\n", filename, errno);
	printf ("0"); /* any output to avoid mkprom fail on the very first run */ 
	return errno;
    }

    if (!bfd_check_format (abfd, bfd_object)) {
	bfd_close(abfd);
	fprintf (stderr, "bfd error in '%s' %d\n", filename, errno);
	return errno;
    }

    if (argc > 2) {
	if (strcmp(argv[2], "-e") == 0) {
	    startAddress = (unsigned int) abfd->start_address;
	    printf("0x%08X", startAddress);
	    bfd_close(abfd);
	    return 0;
	}
	else if (strncmp(argv[2], "-t", 2) == 0) {
	    for (p = abfd->sections; p != NULL; p = p->next) {
		if ((bfd_get_section_flags(abfd, p) & SEC_ALLOC) && (bfd_section_size(abfd, p) > 0)) {
		    bfd_size_type top = bfd_section_vma(abfd, p) + bfd_section_size(abfd, p);
		    if (appEnd < top)
			appEnd = top;
		}
	    }
	    bfd_close(abfd);
	    if (argv[2][2] == '8') {
		if (appEnd % 0x8)
		    appEnd += 0x8 - (appEnd % 0x8);
	    }
	    if (argv[2][2] == '4') {
		if (appEnd % 0x1000)
		    appEnd += 0x1000 - (appEnd % 0x1000);
	    }
	    if ((argv[2][2] == 0) || (argv[2][3] == 0)) {
		printf("0x%08X", appEnd);
		return 0;
	    }

	    if (argc<4) {
		fprintf(stderr, "Missing relative address argument with '%s'\n", argv[2]);
		return -1;
	    }

	    errno = 0;    /* To distinguish success/failure after call */
	    unsigned int readVal = strtoul(argv[3],(char **)NULL,0);
	    if ((errno == ERANGE && (readVal == LONG_MAX || readVal == LONG_MIN))
		|| (errno != 0 && readVal == 0)) {
		fprintf(stderr, "Invalid relative address argument '%s'\n", argv[3]);
		return -1;
	    }

	    if (argv[2][3] == 'a') {
		printf("0x%08X", readVal-appEnd);
		return 0;
	    }
	    if (argv[2][3] == 'b') {
		printf("0x%08X", appEnd-readVal);
		return 0;
	    }
	    fprintf(stderr, "Invalid option '%s'\n", argv[2]);
	    return -1;
	}
	else {
	    fprintf(stderr, "invalid option '%s'\n", argv[2]);
	    bfd_close(abfd);
	    return -1;
	}
    }

    nsec = abfd->section_count;
    sectionaddr = (unsigned int* ) malloc (nsec*sizeof(unsigned int ));
    sectionsize = (unsigned int* ) malloc (nsec*sizeof(unsigned int ));
    sectiondata = (unsigned int**) malloc (nsec*sizeof(unsigned int*));

    for (p = abfd->sections; p != NULL; p = p->next) {
	bfd_vma  base_addr = bfd_section_vma(abfd, p);
	bfd_size_type size = bfd_section_size (abfd, p);
	const char   *name = bfd_section_name(abfd, p);
	flagword     flags = bfd_get_section_flags(abfd, p);

	if ((flags & SEC_ALLOC) && size > 0) {
	    fprintf(stderr, "%s: addr=%p size=%d, flags=%x\n", name, (void*) base_addr, (int) size, flags);

	    if (flags & SEC_LOAD) {
		/* read contents and sort endianness. Add crc */
		unsigned char * readcont = (unsigned char *) malloc(size+4);
		unsigned char * sortcont = (unsigned char *) malloc(size+4);
		bfd_get_section_contents (abfd, p, readcont, 0, size);
		for (i=0; i<size; i++)
		    if (littleend)
			sortcont[i] = readcont[i^3];
		    else
			sortcont[i] = readcont[i];
	    
		unsigned char crc = rmap_crc_calc(readcont, size);
		sortcont[size] = sortcont[size+1] = sortcont[size+2] = sortcont[size+3] = crc;
		free(readcont);
		sectiondata[cursec] = (unsigned int*) sortcont;
	    }
	    else {
		sectiondata[cursec] = (unsigned int*) 0;
	    }

	    sectionaddr[cursec] = base_addr;
	    sectionsize[cursec] = size;
	    cursec++;
	}
	else {
	    fprintf(stderr, "SKIPPING %s: addr=%p size=%d, flags=%x\n", name, (void*) base_addr, (int) size, flags);
	}
    }

    if (cursec == 0) {
	fprintf(stderr, "no loadable section found.\n");
	bfd_close(abfd);
	return 0;
    }

    printf("#define SPWBOOT_NSECTIONS %d\n\n", cursec);

    printf("static const unsigned int spwboot_sectionaddr[] = {\n");
    for (i=0; i<cursec; i++) {
	printf("0x%08X,\n", sectionaddr[i]);
    }
    printf("};\n\n");

    printf("static const unsigned int spwboot_sectionsize[] = {\n");
    for (i=0; i<cursec; i++) {
	printf("0x%08X,\n", sectionsize[i]);
    }
    printf("};\n\n");

    for (i=0; i<cursec; i++) {
	if (sectiondata[i]) {
	    printf("static const unsigned int spwboot_section%d[] = {\n", i);
	    for (n=0; n<sectionsize[i]/4+1; n++) {
		printf("  0x%08X,\n", sectiondata[i][n]);
	    }
	    printf("};\n\n");
	    free(sectiondata[i]);
	}
    }

    printf("static const unsigned int* spwboot_sectiondata[] = {\n");
    for (i=0; i<cursec; i++) {
	if (sectiondata[i])
	    printf("  spwboot_section%d,\n", i);
	else
	    printf("  0x0,\n");
    }
    printf("};\n\n");

    free(sectionaddr);
    free(sectionsize);
    free(sectiondata);

    bfd_close(abfd);
    return 0;
}
