/*****************************************************************************/
/*   This file is a part of the GRLIB VHDL IP LIBRARY */
/*   Copyright (C) 2004 GAISLER RESEARCH */

/*   This program is free software; you can redistribute it and/or modify */
/*   it under the terms of the GNU General Public License as published by */
/*   the Free Software Foundation; either version 2 of the License, or */
/*   (at your option) any later version. */

/*   See the file COPYING for the full details of the license. */
/*****************************************************************************/

#include "spwapi.h"
#include <stdlib.h>
#include <stdio.h>

static char *almalloc(int sz)
{
  char *tmp;
  tmp = calloc(1,2*sz);
  tmp = (char *) (((int)tmp+sz) & ~(sz -1));
  return(tmp);
}

static inline int loadmem(int addr)
{
  int tmp;        
  asm(" lda [%1]1, %0 "
      : "=r"(tmp)
      : "r"(addr)
    );
  return tmp;
}

/* static void storemem(int addr, int data)  */
/* { */
/*         asm("sta %0, [%1]1 " */
/*             :  */
/*             : "r"(data), "r"(addr)  */
/*            ); */
/* } */



unsigned char rmap_crc_calc(unsigned char *data, unsigned int len)
{
    unsigned char crc = 0;
    unsigned int i;
    for (i = 0; i < len; i++) {
	crc = RMAP_CRCTable[(crc ^ data[i]) & 0xff];
    }
    return crc;
}

int build_rmap_hdr(struct rmap_pkt *pkt, char *hdr, int *size, int softCrc)
{
    int i;
    int j;
    int type;
    int write;
    int srcspalen;
    if ((pkt->type != readcmd) && (pkt->type != writecmd) && (pkt->type != rmwcmd) &&
	(pkt->type != readrep) && (pkt->type != writerep) && (pkt->type != rmwrep)) {
	return 1;
    }
    if ((pkt->verify != yes) && (pkt->verify != no)) {
	return 1;
    }
    if ((pkt->ack != yes) && (pkt->ack != no)) {
	return 2;
    }
    if ((pkt->incr != yes) && (pkt->incr != no)) {
	return 3;
    }
    if ( (pkt->dstspalen < 0) || (pkt->dstspalen > 228) ) {
	return 4;
    }
    if ( (pkt->srcspalen < 0) || (pkt->srcspalen > 12) ) {
	return 5;
    }
    if( (pkt->destkey < 0) || (pkt->destkey > 255) ) {
	return 6;
    }
    if( (pkt->destaddr < 0) || (pkt->destaddr > 255) ) {
	return 7;
    }
    if( (pkt->srcaddr < 0) || (pkt->srcaddr > 255) ) {
	return 9;
    }
    if( (pkt->tid < 0) || (pkt->tid > 65535) ) {
	return 10;
    }
    if( (pkt->len < 0) || (pkt->len > 16777215) ) {
	return 11;
    }
    if( (pkt->status < 0) || (pkt->status > 12) ) {
	return 12;
    }
    if ((pkt->type == writecmd) || (pkt->type == writerep)) {
	write = 1;
    } else {
	write = 0;
    }
    if ((pkt->type == writecmd) || (pkt->type == readcmd) || (pkt->type == rmwcmd)) {
	type = 1;
	*size = pkt->dstspalen + 15;
	srcspalen = pkt->srcspalen/4;
	if ( (pkt->srcspalen % 4) != 0) {
	    srcspalen = srcspalen + 1;
	}
	*size = srcspalen * 4 + *size;
	for(i = 0; i < pkt->dstspalen; i++) {
	    hdr[i] = pkt->dstspa[i];
	}
	hdr[pkt->dstspalen] = (char)pkt->destaddr;
	hdr[pkt->dstspalen+1] = (char)0x01;
	hdr[pkt->dstspalen+2] = (char)0;
	hdr[pkt->dstspalen+2] = hdr[pkt->dstspalen+2] | (type << 6) | 
	    (write << 5) | (pkt->verify << 4) | (pkt->ack << 3) | (pkt->incr << 2) | srcspalen;
	hdr[pkt->dstspalen+3] = (char)pkt->destkey;
	j = 0;
	for(i = 0; i < srcspalen*4; i++) {
	    if ((srcspalen*4-i) > (pkt->srcspalen)) {
		hdr[pkt->dstspalen+4+i] = 0x00;
	    } else {
		hdr[pkt->dstspalen+4+i] = pkt->srcspa[j];
		j++;
	    }
                        
	}
	hdr[pkt->dstspalen+4+srcspalen*4] = (char)pkt->srcaddr;
	hdr[pkt->dstspalen+5+srcspalen*4] = (char)((pkt->tid >> 8) & 0xFF);
	hdr[pkt->dstspalen+6+srcspalen*4] = (char)(pkt->tid & 0xFF);
	hdr[pkt->dstspalen+7+srcspalen*4] = (char)0;
	hdr[pkt->dstspalen+8+srcspalen*4] = (char)((pkt->addr >> 24) & 0xFF);
	hdr[pkt->dstspalen+9+srcspalen*4] = (char)((pkt->addr >> 16) & 0xFF);
	hdr[pkt->dstspalen+10+srcspalen*4] = (char)((pkt->addr >> 8) & 0xFF);
	hdr[pkt->dstspalen+11+srcspalen*4] = (char)(pkt->addr & 0xFF);
	hdr[pkt->dstspalen+12+srcspalen*4] = (char)((pkt->len >> 16) & 0xFF);
	hdr[pkt->dstspalen+13+srcspalen*4] = (char)((pkt->len >> 8) & 0xFF);
	hdr[pkt->dstspalen+14+srcspalen*4] = (char)(pkt->len & 0xFF);
    } else {
	type = 0;
	if (pkt->type == writerep) {
	    *size = pkt->srcspalen + 7;
	} else {
	    *size = pkt->srcspalen + 11;
	}
	srcspalen = pkt->srcspalen/4;
	if ( (pkt->srcspalen % 4) != 0) {
	    srcspalen = srcspalen + 1;
	}
	for(i = 0; i < pkt->srcspalen; i++) {
	    hdr[i] = pkt->srcspa[i];
	}
	hdr[pkt->srcspalen] = (char)pkt->srcaddr;
	hdr[pkt->srcspalen+1] = (char)0x01;
	hdr[pkt->srcspalen+2] = (char)0;
	hdr[pkt->srcspalen+2] = hdr[pkt->srcspalen+2] | (type << 6) | 
	    (write << 5) | (pkt->verify << 4) | (pkt->ack << 3) | (pkt->incr << 2) | srcspalen;
	hdr[pkt->srcspalen+3] = (char)pkt->status;
	hdr[pkt->srcspalen+4] = (char)pkt->destaddr;
	hdr[pkt->srcspalen+5] = (char)((pkt->tid >> 8) & 0xFF);
	hdr[pkt->srcspalen+6] = (char)(pkt->tid & 0xFF);
	if (pkt->type != writerep) {
	    hdr[pkt->srcspalen+7] = (char)0;
	    hdr[pkt->srcspalen+8] = (char)((pkt->len >> 16) & 0xFF);
	    hdr[pkt->srcspalen+9] = (char)((pkt->len >> 8) & 0xFF);
	    hdr[pkt->srcspalen+10] = (char)(pkt->len & 0xFF);
	}
    }
    if (softCrc) {
	hdr[*size] = rmap_crc_calc(hdr + pkt->dstspalen, *size - pkt->dstspalen);
	*size = *size + 1;
    }
    return 0;
}

int spw_setparam(int nodeaddr, int clkdiv, int destkey,
                 int timetxen, int timerxen, int spwadr, 
                 int khz, struct spwvars *spw, int port,
                 int clkdivs) 
{
        if ((nodeaddr < 0) || (nodeaddr > 255)) {
                return 1;
        }
        if ((clkdiv < 0) || (clkdiv > 255)) {
                return 1;
        }
        if ((clkdivs < 0) || (clkdivs > 255)) {
                return 1;
        }
        if ((destkey < 0) || (destkey > 255)) {
                return 1;
        }
        if ((timetxen < 0) || (timetxen > 1)) {
                return 1;
        }
        if ((timerxen < 0) || (timerxen > 1)) {
                return 1;
        }
        spw->timetxen = timetxen;
        spw->timerxen = timerxen;
        spw->destkey = destkey;
        spw->nodeaddr = nodeaddr;
        spw->clkdiv = clkdiv;
        spw->clkdivs = clkdivs;
        spw->khz = khz;
	spw->port = port;
        spw->regs = (struct spwregs *) spwadr;
        return 0;
}

int spw_setparam_dma(int dmachan, int addr, int mask, int nospill, int rxmaxlen, struct spwvars *spw) 
{
        if ((addr < 0) || (addr > 255)) {
                return 1;
        }
        if ((mask < 0) || (mask > 255)) {
                return 1;
        }
        if ((rxmaxlen < 0) || (rxmaxlen > 33554432)) {
                return 1;
        }
        if ((nospill < 0) || (nospill > 1)) {
                return 1;
        }
        spw->dma[dmachan].nospill = nospill;
        spw->dma[dmachan].addr = addr;
        spw->dma[dmachan].mask = mask;
        spw->dma[dmachan].rxmaxlen = rxmaxlen;
        return 0;
}


int spw_init(struct spwvars *spw)
{
        int i;
        int j;
        int tmp;
        /*determine grspw version by checking if timer and disconnect register exists */
        spw->regs->timer = 0xFFFFFF;
        tmp = loadmem((int)&(spw->regs->timer));
        spw->ver = 0;
        if (!tmp) 
                spw->ver = 1;
        tmp = loadmem((int)&(spw->regs->ctrl));
        spw->rmap = (tmp >> 31) & 1;
        spw->rxunaligned = (tmp >> 30) & 1;
        spw->rmapcrc = (tmp >> 29) & 1;
        spw->dmachan = ((tmp >> 27) & 3) + 1;
        /*reset core */
        spw->regs->ctrl = (1 << 6); 
        tmp = loadmem((int)&(spw->regs->ctrl));
        while ((tmp >> 6) & 1) {
                tmp = loadmem((int)&(spw->regs->ctrl));
        }
        spw->regs->nodeaddr = spw->nodeaddr; /*set node address*/
        spw->regs->clkdiv = spw->clkdiv | (spw->clkdivs << 8);  /* set clock divisor */

        for(i = 0; i < spw->dmachan; i++) {
                spw->regs->dma[i].rxmaxlen = spw->dma[i].rxmaxlen; /*set rx maxlength*/
                if (loadmem((int)&(spw->regs->dma[i].rxmaxlen)) != spw->dma[i].rxmaxlen) {
                        return 1;
                }
        }
        if (spw->khz) 
                spw->regs->timer = ((spw->khz*64 + 9999)/10000) | ((((spw->khz*850 + 999999)/1000000)-3) << 12);
        if (spw->rmap == 1) {
                spw->regs->destkey = spw->destkey;
        }
        for(i = 0; i < spw->dmachan; i++) {
                spw->regs->dma[i].ctrl = 0xFFFE01E0; /*clear status, set ctrl for dma chan*/
                if (loadmem((int)&(spw->regs->dma[i].ctrl)) != 0) {
                        return 2;
                }
                /* set tx descriptor pointer*/
                if ((spw->dma[i].txd = (struct txdescriptor *)almalloc(1024)) == NULL) {
                        return 3;
                }
                spw->dma[i].txpnt = 0;
                spw->dma[i].txchkpnt = 0;
                spw->regs->dma[i].txdesc = (int) spw->dma[i].txd;
                /* set rx descriptor pointer*/
                if (( spw->dma[i].rxd = (struct rxdescriptor *)almalloc(1024)) == NULL) {
                        return 4;
                }
                spw->dma[i].rxpnt = 0;
                spw->dma[i].rxchkpnt = 0;
                spw->regs->dma[i].rxdesc = (int) spw->dma[i].rxd;
        }
        spw->regs->status = 0xFFF; /*clear status*/
        spw->regs->ctrl = 0x2 | (spw->timetxen << 10) | (spw->timerxen << 11) | (spw->port << 21); /*set ctrl*/
        for(i = 0; i < spw->dmachan; i++) {
                spw->regs->dma[i].ctrl = loadmem((int)&(spw->regs->dma[i].ctrl)) | (spw->dma[i].nospill << 12);
        }
        return 0;
}

int spw_linkStatus(struct spwvars *spw)
{
    int tmp;
    tmp = loadmem((int)&(spw->regs->status));
    return (tmp >> 21) & 0x7;
}

int wait_running(struct spwvars *spw) 
{
        int i;
        int j;
        
        j = 0;
        while (((loadmem((int)&(spw->regs->status)) >> 21) & 7) != 5) {
                if (j > 1000) {
                        return 1;
                }
                for(i = 0; i < 1000; i++) {}
		j++;
        }
        return 0;
}

int set_txdesc(int dmachan, int pnt, struct spwvars *spw) 
{
        spw->regs->dma[dmachan].txdesc = pnt;
        spw->dma[dmachan].txpnt = 0;
        spw->dma[dmachan].txchkpnt = 0;
        if (loadmem((int)&(spw->regs->dma[dmachan].txdesc)) != pnt) {
                return 1;
        }
        return 0;
}

int set_rxdesc(int dmachan, int pnt, struct spwvars *spw) 
{
        spw->regs->dma[dmachan].rxdesc = pnt;
        spw->dma[dmachan].rxpnt = 0;
        spw->dma[dmachan].rxchkpnt = 0;
        if (loadmem((int)&(spw->regs->dma[dmachan].rxdesc)) != pnt) {
                return 1;
        }
        return 0;
}

void spw_disable(struct spwvars *spw) 
{
        spw->regs->ctrl = loadmem((int)&(spw->regs->ctrl)) | 1;
}

void spw_enable(struct spwvars *spw) 
{
        spw->regs->ctrl = loadmem((int)&(spw->regs->ctrl)) & 0x220F7E;
}

void spw_start(struct spwvars *spw) 
{
        spw->regs->ctrl = loadmem((int)&(spw->regs->ctrl)) | (1 << 1);
}

void spw_stop(struct spwvars *spw) 
{
        spw->regs->ctrl = loadmem((int)&(spw->regs->ctrl)) & 0x220F7D;
}

int spw_setclockdiv(struct spwvars *spw) 
{
        if ( (spw->clkdiv < 0) || (spw->clkdiv > 255) ) {
                return 1;
        } else {
                spw->regs->clkdiv = spw->clkdiv;
                return 0;
        }
}

int spw_set_nodeadr(struct spwvars *spw) 
{
        if ( (spw->nodeaddr < 0) || (spw->nodeaddr > 255) || 
             (spw->mask < 0) || (spw->mask > 255) ) {
                return 1;
        } else {
                spw->regs->nodeaddr = (spw->nodeaddr & 0xFF) | ((spw->mask & 0xFF) << 8);
                return 0;
        }
}

int spw_set_chanadr(int dmachan, struct spwvars *spw) 
{
        if ( (spw->dma[dmachan].addr < 0) || (spw->dma[dmachan].addr > 255) || 
             (spw->dma[dmachan].mask < 0) || (spw->dma[dmachan].mask > 255) ) {
                return 1;
        } else {
                spw->regs->dma[dmachan].addr = (spw->dma[dmachan].addr & 0xFF) | ((spw->dma[dmachan].mask & 0xFF) << 8);
                return 0;
        }
}

int spw_set_rxmaxlength(int dmachan, struct spwvars *spw) 
{
        if ((spw->dma[dmachan].rxmaxlen < 4) || (spw->dma[dmachan].rxmaxlen > 33554431)) {
                return 1;
        } else {
                spw->regs->dma[dmachan].rxmaxlen = spw->dma[dmachan].rxmaxlen;
                return 0;
        }
}

int spw_tx(int dmachan, int hcrc, int dcrc, int skipcrcsize, int hsize, char *hbuf, int dsize, char *dbuf, struct spwvars *spw /*, int softCrc */)
{
        if ((dsize < 0) || (dsize > 16777215)) {
                return 6;
        }
        if ((hsize < 0) || (hsize > 255)) {
                return 5;
        }
        if ((dbuf == NULL) || (hbuf == NULL)) {
                return 4;
        }
        if ( (((hcrc == 1) || (dcrc == 1)) && ((spw->rmapcrc | spw->rmap) == 0)) || (hcrc < 0) || (hcrc > 1) || (dcrc < 0) || (dcrc > 1)) {
                return 3;
        } 
        if ((skipcrcsize < 0) || (skipcrcsize > 15) ) {
                return 2;
        }
        if ((loadmem((int)&(spw->dma[dmachan].txd[spw->dma[dmachan].txpnt].ctrl)) >> 12) & 1) {
                return 1;
        }
	/* if (softCrc) { */
	/*     dbuf[dsize] = rmap_crc_calc(dbuf, dsize); */
	/*     dsize++; */
	/* } */

        spw->dma[dmachan].txd[spw->dma[dmachan].txpnt].haddr = (int)hbuf;
        spw->dma[dmachan].txd[spw->dma[dmachan].txpnt].dlen = dsize;
        spw->dma[dmachan].txd[spw->dma[dmachan].txpnt].daddr = (int)dbuf;
        if (spw->dma[dmachan].txpnt == 63) {
                spw->dma[dmachan].txd[spw->dma[dmachan].txpnt].ctrl = 0x3000 | hsize | (hcrc << 16) | (dcrc << 17) | (skipcrcsize << 8);
                spw->dma[dmachan].txpnt = 0;
        } else {
                spw->dma[dmachan].txd[spw->dma[dmachan].txpnt].ctrl = 0x1000 | hsize | (hcrc << 16) | (dcrc << 17) | (skipcrcsize << 8);
                spw->dma[dmachan].txpnt++;
        }
        
        spw->regs->dma[dmachan].ctrl = loadmem((int)&(spw->regs->dma[dmachan].ctrl)) & 0xFAAA | 1;
        
        return 0;
}

int spw_rx(int dmachan, char *buf, struct spwvars *spw) 
{
        if (((loadmem((int)&(spw->dma[dmachan].rxd[spw->dma[dmachan].rxpnt].ctrl)) >> 25) & 1)) {
                return 1;
        }
        spw->dma[dmachan].rxd[spw->dma[dmachan].rxpnt].daddr = (int)buf;
        if (spw->dma[dmachan].rxpnt == 127) {
                spw->dma[dmachan].rxd[spw->dma[dmachan].rxpnt].ctrl = 0x6000000;
                spw->dma[dmachan].rxpnt = 0;
        } else {
                spw->dma[dmachan].rxd[spw->dma[dmachan].rxpnt].ctrl = 0x2000000;
                spw->dma[dmachan].rxpnt++;
        }
        spw->regs->dma[dmachan].ctrl = loadmem((int)&(spw->regs->dma[dmachan].ctrl)) & 0xF955 | 2 | (1 << 11);
        return 0;
}

int spw_checkrx(int dmachan, int *size, struct rxstatus *rxs, struct spwvars *spw) 
{
        int tmp;
        tmp = loadmem((int)&(spw->dma[dmachan].rxd[spw->dma[dmachan].rxchkpnt].ctrl));
        if (!((tmp >> 25) & 1)) {
                *size = tmp & 0x1FFFFFF;
                rxs->truncated = (tmp >> 31) & 1;
                rxs->dcrcerr = (tmp >> 30) & 1;
                rxs->hcrcerr = (tmp >> 29) & 1;
                rxs->eep = (tmp >> 28) & 1;
                if (spw->dma[dmachan].rxchkpnt == 127) {
                        spw->dma[dmachan].rxchkpnt = 0;
                } else {
                        spw->dma[dmachan].rxchkpnt++;
                }
                return 0;
        } else {
                return 1;
        }
}

int spw_checktx(int dmachan, struct spwvars *spw)
{
        int tmp;
        tmp = loadmem((int)&(spw->dma[dmachan].txd[spw->dma[dmachan].txchkpnt].ctrl));
        if (!((tmp >> 12) & 1)) {
                if (spw->dma[dmachan].txchkpnt == 63) {
                        spw->dma[dmachan].txchkpnt = 0;
                } else {
                        spw->dma[dmachan].txchkpnt++;
                }
                if ((tmp >> 15) & 1) {
                        return 2;
                } else {
                        return 1;
                }
        } else {
                return 0;
        }
}

void send_time(struct spwvars *spw)
{
        int i;
        while( ((loadmem((int)&(spw->regs->ctrl)) >> 4) & 1)) {
                for(i = 0; i < 16; i++) {}
        }
        spw->regs->ctrl = loadmem((int)&(spw->regs->ctrl)) | (1 << 4);
}

int send_time_exp(int ctrl, int time, struct spwvars *spw)
{
        int i;
        int code;
        if ((time < 0) || (time > 63)) {
                printf("Illegal time-value");
                return -1;
        }
        if (time == 0) {
                time = 63;
        } else {
                time = time - 1;
        }
        code = ((ctrl << 6) & 0xC0) | time;
        
        while( ((loadmem((int)&(spw->regs->ctrl)) >> 4) & 1)) {
                for(i = 0; i < 16; i++) {}
        }
        spw->regs->timereg = (loadmem((int)&(spw->regs->timereg)) & 0xFFFFFF00) | (0xFF & code);
        spw->regs->ctrl = loadmem((int)&(spw->regs->ctrl)) | (1 << 4);
        return 0;
}

int check_time(struct spwvars *spw) 
{
        int tmp = loadmem((int)&(spw->regs->status)) & 1;
        if (tmp) {
                spw->regs->status = loadmem((int)&(spw->regs->status)) | 1;
        }
        return tmp;
}

int get_time(struct spwvars *spw) 
{
        return (loadmem((int)&(spw->regs->timereg)) & 0x3F );
}

int get_time_ctrl(struct spwvars *spw) 
{
        return ((loadmem((int)&(spw->regs->timereg)) >> 6) & 0x3);
}

void spw_reset(struct spwvars *spw) 
{
  spw->regs->ctrl = loadmem((int)&(spw->regs->ctrl)) | (1 << 6);
}
        
void spw_rmapen(struct spwvars *spw) 
{
  spw->regs->ctrl = loadmem((int)&(spw->regs->ctrl)) | (1 << 16);
}

void spw_rmapdis(struct spwvars *spw) 
{
  spw->regs->ctrl = loadmem((int)&(spw->regs->ctrl)) & 0x2EFFFF;
}

int spw_setdestkey(struct spwvars *spw) 
{
  if ((spw->destkey < 0) || (spw->destkey > 255)) {
    return 1;
  }
  spw->regs->destkey = spw->destkey;
  return 0;
}

void spw_setsepaddr(int dmachan, struct spwvars *spw) 
{
        spw->regs->dma[dmachan].ctrl = loadmem((int)&(spw->regs->dma[dmachan].ctrl)) | (1 << 13);
}

void spw_disablesepaddr(int dmachan, struct spwvars *spw) 
{
        spw->regs->dma[dmachan].ctrl = loadmem((int)&(spw->regs->dma[dmachan].ctrl)) & 0xFFFFDFFF;
}


void spw_enablerx(int dmachan, struct spwvars *spw) 
{
        spw->regs->dma[dmachan].ctrl = loadmem((int)&(spw->regs->dma[dmachan].ctrl)) | 0x2;
}


void spw_disablerx(int dmachan, struct spwvars *spw) 
{
        spw->regs->dma[dmachan].ctrl = loadmem((int)&(spw->regs->dma[dmachan].ctrl)) & 0xFFFFFFFD;
}

void spw_disable_promiscuous(struct spwvars *spw) 
{
        spw->regs->ctrl = loadmem((int)&(spw->regs->ctrl)) & 0xFFFFFFDF;
}

void spw_enable_promiscuous(struct spwvars *spw) 
{
        spw->regs->ctrl = loadmem((int)&(spw->regs->ctrl)) | (1 << 5);
}

