------------------------------------------------------------------------------
--  This file is a part of the GRLIB VHDL IP LIBRARY
--  Copyright (C) 2003 - 2008, Gaisler Research
--  Copyright (C) 2008 - 2014, Aeroflex Gaisler
--  Copyright (C) 2015 - 2023, Cobham Gaisler
--  Copyright (C) 2023 - 2025, Frontgrade Gaisler
--
--  This program is free software; you can redistribute it and/or modify
--  it under the terms of the GNU General Public License as published by
--  the Free Software Foundation; version 2.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--  GNU General Public License for more details.
--
--  You should have received a copy of the GNU General Public License
--  along with this program; if not, write to the Free Software
--  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
-----------------------------------------------------------------------------
-- Entity:      axis_gearbox
-- File:        axis_gearbox.vhd
-- Author:      Carl Ehrenstrahle - Frontgrade Gaisler AB
-- Description: AXI-Stream Gearbox
-- Combines or splits vectors using the AXI-Stream protocol.
-- In case of combination, a partial last result can be generated by asserting
-- tlast. When splitting the last created output will carry the last flag.
-- The gearbox works on the lowest signifcant bits first.
-- Implementeded as a circular buffer with an output register there are two
-- clock cycles worth of latency through the gearbox.
------------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;

library grlib;
use grlib.config_types.all;
use grlib.config.all;

entity axis_gearbox is
  generic (
    wl_in : positive := 1;
    wl_out : positive := 1
  );
  port (
    clk : in std_logic;
    rst : in std_logic;

    s_axis_tdata : in std_logic_vector(wl_in - 1 downto 0);
    s_axis_tlast : in std_logic;
    s_axis_tvalid : in std_logic;
    s_axis_tready : out std_logic;

    m_axis_tdata : out std_logic_vector(wl_out - 1 downto 0);
    m_axis_tlast : out std_logic;
    m_axis_tvalid : out std_logic;
    m_axis_tready : in std_logic
  );
end entity;

architecture rtl of axis_gearbox is
  function to_sl (a : boolean) return std_logic is
  begin
    if a then
      return '1';
    else
      return '0';
    end if;
  end function;

  constant ASYNC_RESET : boolean :=
    GRLIB_CONFIG_ARRAY(grlib_async_reset_enable) = 1;

  constant buf_len_c : positive := wl_out + wl_in;

  type reg_t is record
    out_dv : std_logic;
    out_data : std_logic_vector(wl_out - 1 downto 0);
    out_last : std_logic;
    buf_data : std_logic_vector(buf_len_c - 1 downto 0);
    rd_ptr : integer range 0 to wl_in + wl_out - 1;
    rd_ptr_lap : std_logic;
    wr_ptr : integer range 0 to wl_in + wl_out - 1;
    wr_ptr_lap : std_logic;
    has_last : std_logic;
  end record;

  constant reg_reset_c : reg_t := (
    out_dv => '0',
    out_data => (others => '0'),
    out_last => '0',
    buf_data => (others => '0'),
    rd_ptr => 0,
    rd_ptr_lap => '0',
    wr_ptr => 0,
    wr_ptr_lap => '0',
    has_last => '0'
  );

  signal r, rin : reg_t;
begin

  --------------------
  -- Combinatorial
  --------------------
  passthrough_gen : if wl_in = wl_out generate
    m_axis_tdata <= s_axis_tdata;
    m_axis_tlast <= s_axis_tlast;
    m_axis_tvalid <= s_axis_tvalid;
    s_axis_tready <= m_axis_tready;
  end generate;
  gearbox_gen : if wl_in /= wl_out generate
    comb_p : process(r, m_axis_tready, s_axis_tdata, s_axis_tvalid, s_axis_tlast)
      variable v : reg_t;
      -- ptr calc
      variable wrapped : boolean;
      variable t_wr_ptr : integer range 0 to 2 * buf_len_c;
      variable level : integer range 0 to wl_in + wl_out;
      -- decisions
      variable do_read : boolean;
      variable can_write : boolean;
      -- writing
      variable twr : integer range 0 to wl_in - 1;
      -- reading
      variable trd : integer range 0 to wl_out - 1;
      variable last_flush : boolean;
    begin
      v := r; -- Copy current state.

      -- Calculate level
      wrapped := (r.wr_ptr_lap xor r.rd_ptr_lap) = '1';
      if wrapped then
        t_wr_ptr := r.wr_ptr + buf_len_c;
      else
        t_wr_ptr := r.wr_ptr;
      end if;
      level := t_wr_ptr - r.rd_ptr;

      -- Last flushing handling.
      last_flush := false;
      if r.has_last = '1' and level <= wl_out and (m_axis_tready = '1' or r.out_dv = '0') then
        last_flush := true;
      end if;

      do_read := (m_axis_tready = '1' or r.out_dv = '0') and ((level >= wl_out) or last_flush);

      -- Reading frees up an output word's worth of space.
      if do_read then
        can_write := ((buf_len_c + wl_out) - level >= wl_in);
      else
        can_write := (buf_len_c - level >= wl_in);
      end if;
      -- Wait until last has been flushed until allowing further writes.
      can_write := can_write and r.has_last = '0';

      -- Reading
      if do_read then
        -- Pointer update
        if r.rd_ptr + wl_out >= buf_len_c then
          -- Pointer is wrapping, so turn it around.
          v.rd_ptr := (r.rd_ptr + wl_out) - buf_len_c;
          v.rd_ptr_lap := not r.rd_ptr_lap; -- passed a lap of the buffer
        else
          v.rd_ptr := r.rd_ptr + wl_out;
        end if;

        -- Output data assignment
        if r.rd_ptr + wl_out > buf_len_c then
          trd := buf_len_c - r.rd_ptr;

          v.out_data(trd - 1 downto 0) := r.buf_data(buf_len_c - 1 downto r.rd_ptr);
          v.out_data(v.out_data'high downto trd) := r.buf_data(wl_out - trd - 1 downto 0);
        else
          v.out_data := r.buf_data(r.rd_ptr + wl_out - 1 downto r.rd_ptr);
        end if;
      end if;

      -- If pipeline can move then update output validity with read status.
      if m_axis_tready = '1' or r.out_dv = '0' then
        v.out_dv := to_sl(do_read);
        v.out_last := to_sl(last_flush);
      end if;

      -- Writing
      if can_write and s_axis_tvalid = '1' then
        -- Pointer update
        if r.wr_ptr + wl_in >= buf_len_c then
          -- Pointer is wrapping, so turn it around.
          v.wr_ptr := (r.wr_ptr + wl_in) - buf_len_c;
          v.wr_ptr_lap := not r.wr_ptr_lap; -- passed a lap of the buffer
        else
          v.wr_ptr := r.wr_ptr + wl_in;
        end if;

        -- Last flag handling.
        v.has_last := s_axis_tlast;

        -- Muxing
        if r.wr_ptr + wl_in > buf_len_c then
          twr := buf_len_c - r.wr_ptr;

          v.buf_data(buf_len_c - 1 downto r.wr_ptr) := s_axis_tdata(twr - 1 downto 0);
          v.buf_data(wl_in - twr - 1 downto 0) := s_axis_tdata(s_axis_tdata'high downto twr);
        else
          v.buf_data(r.wr_ptr + wl_in - 1 downto r.wr_ptr) := s_axis_tdata;
        end if;
      end if;

      -- Do a reset of the control signals due to flush.
      if last_flush then
        v.has_last := '0';
        v.wr_ptr := 0;
        v.wr_ptr_lap := '0';
        v.rd_ptr := 0;
        v.rd_ptr_lap := '0';
      end if;

      rin <= v; -- Commit next state.

      -- Output --
      m_axis_tdata <= r.out_data;
      m_axis_tvalid <= r.out_dv;
      m_axis_tlast <= r.out_last;

      s_axis_tready <= to_sl(can_write);
    end process;
  end generate;

  --------------------
  -- Clocked process
  --------------------
  syncregs : if not ASYNC_RESET generate
    regs_p : process(clk)
    begin
      if rising_edge(clk) then
        r <= rin;
        if rst = '0' then
          r <= reg_reset_c;
        end if;
      end if;
    end process;
  end generate;

  asyncregs : if ASYNC_RESET generate
    regs_p : process(rst, clk)
    begin
      if rst = '0' then
        r <= reg_reset_c;
      elsif rising_edge(clk) then
        r <= rin;
      end if;
    end process;
  end generate;
end architecture;
