#!/usr/bin/env tclsh

set util_rev 1.1
set USAGE \
"This is a script for generating BCH check bytes to arbitrary input data.
Usage: $argv0 \
        \[-elf\]\
        \[-a ADDRESS \]\
        SOURCE \
        SOURCE \
        DEST \
        SIZE
  SOURCE  name of the input binary file
  DEST    name of the output binary file which will contain the SOURCE binary
          appended with calculated BCH check bytes.
  SIZE    target ROM size, in KiB."

set OBJCOPY sparc-gaisler-elf-objcopy

# Convert number into 32-bit hex string.
proc tohex32 {v} {
        set v [expr {$v & 0xffffffff}]
        format "0x%08x" $v
}

# Calculate bch test check bits for a 32-bit word
proc bch {word} {
        lappend bits_arrays [list 0 4 6 7 8 9 11 14 17 18 19 21 26 28 29 31]
        lappend bits_arrays [list 0 1 2 4 6 8 10 12 16 17 18 20 22 24 26 28]
        lappend bits_arrays [list 0 3 4 7 9 10 13 15 16 19 20 23 25 26 29 31]
        lappend bits_arrays [list 0 1 5 6 7 11 12 13 16 17 21 22 23 27 28 29]
        lappend bits_arrays [list 2 3 4 5 6 7 14 15 18 19 20 21 22 23 30 31]
        lappend bits_arrays [list 8 9 10 11 12 13 14 15 24 25 26 27 28 29 30 31]
        lappend bits_arrays [list 0 1 2 3 4 5 6 7 24 25 26 27 28 29 30 31]
        set negs [list 0 0 1 1 0 0 0]

        for {set i 0} {$i < 32} {incr i} {
                lappend d [expr {($word >> $i) & 1}]
        }
        set ret 0
        foreach bits_array $bits_arrays neg $negs {
                set cb $neg
                foreach bit $bits_array {
                        set cb [expr {$cb ^ [lindex $d $bit]}]
                }
                set ret [expr {($cb << 6) | ($ret >> 1)}]
                lappend cbs $cb
        }
        return [format "0x%02x" $ret]
}

# Given a list of words, return a list of bch check bytes
proc bchlist {words} {
	lappend cbs
	foreach word $words {
		set data [tohex32 $word]
		set cb [bch $data]
		lappend cbs $cb
	}
	return $cbs
}

# Write integer as 32-bit big-endian binary bytes to channel.
proc w32 {ch value} { puts -nonewline $ch [binary format I $value] }
# Write integer as 8-bit big-endian binary bytes to channel.
proc w8 {ch value} { puts -nonewline $ch [binary format c $value] }


### Start of script ###

# Parse optional arguments
set state flag
set opt_elf false
set fns [list]
set ASM_ADDR 0
foreach arg [lrange $argv 0 end] {
        switch -- $state {
                flag {
                        switch -glob -- $arg {
                                -elf {set opt_elf true}
                                -a  {set state a}
                                -* {puts $USAGE; exit 1}
                                default {lappend fns $arg}
                        }
                }
                a {
                        set ASM_ADDR $arg
                        set state flag
                }
        }
}
unset state


# - Check command line parameters
if {[llength $fns] < 3 } {
        puts $USAGE
        exit 1
}

set SOURCE [lindex $fns 0]
set DEST [lindex $fns 1]
set SIZE [lindex $fns 2]
set SIZEB [expr {$SIZE * 1024}]
# SIZE is size in KiB. SIZEB is size in B.

# Limit the output file a bit. However, a 256 MiB binary is still
# unpractical...
if {[expr {256*1024}] < $SIZE} {
	puts "ERROR: SIZE=$SIZE is not supported."
        puts $USAGE
        exit 1
}

# - Open source file and read it.
set sch [open $SOURCE "rb"]
set rawdata [read $sch]
close $sch

# - Add some extra bytes to source to align trailing bytes.
set rawdata "${rawdata}\0\0\0"
binary scan $rawdata I* words
# Fill up list to be a multiple of 16 words
# => bch will be a multiple of 4 words
while {[expr {[llength $words] % 16}]} {
        lappend words 0x50414421
}
set nwords [llength $words]

# - Verify that SOURCE + TCB can fit in SIZEB
if {$SIZEB < ($nwords*4 + $nwords)} {
	puts "Source file $SOURCE plus test check bits do not fit inside $SIZE KiB."
	exit 1
}

# - Generate list of TCB from data word in SOURCE.
set bchs [bchlist $words]

proc output_bin {} {
        global SOURCE
        global DEST
        global SIZEB
        global nwords
        global words
        global bchs
        # - Open destination file.
        set dch [open $DEST "wb"]

        # - Copy from source data words to DEST file.
        puts "Writing to DEST ($DEST)..."
        set lastdata [expr {$nwords*4-1}]
        puts [format "0x%08x - 0x%08x : SOURCE ($SOURCE)" 0 $lastdata]
        foreach w $words {
                w32 $dch $w
        }

        # - Calculate start offset of TCB area (tcbstart)
        set firstpad [expr {$lastdata+1}]
        set tcbstart [expr {$SIZEB - $nwords}]
        set lastpad [expr {$tcbstart-1}]

        # - Fill DEST with zeroes until tcbstart
        puts [format "0x%08x - 0x%08x : 0x00 ..." $firstpad $lastpad]
        for {set i [expr {$nwords*4}]} {$i < $tcbstart} {incr i} {
                w8 $dch 0
        }

        # - Copy content of the TCB list to DEST file, "backwards".
        puts [format "0x%08x - 0x%08x : BCH check bytes" $tcbstart [expr {$SIZEB - 1}]]
        foreach b [lreverse $bchs] {
                w8 $dch $b
        }

        close $dch

        if {$SIZEB != [file size $DEST]} {
                puts "ERROR: Output file has unexpected size"
                exit 1
        }
}

proc output_elf {} {
        global OBJCOPY
        global ASM_ADDR
        global SOURCE
        global DEST
        global SIZEB
        global nwords
        global words
        global bchs

        # - Dump DATA
        set addr_data [tohex32 [expr {$ASM_ADDR}]]

        set datafn "${DEST}.raw.data"
        set dch [open $datafn "wb"]

        puts "Writing DATA to $datafn..."
        set lastdata [tohex32 [expr {$addr_data + $nwords*4-1}]]
        puts [format "0x%08x - 0x%08x : $datafn ($SOURCE)" $addr_data $lastdata]
        foreach w $words {
                w32 $dch $w
        }
        close $dch


        # - Dump BCH
        set tcbstart [tohex32 [expr {$SIZEB - $nwords}]]
        set addr_bch  [tohex32 [expr {$ASM_ADDR + $tcbstart}]]

        set bchfn "${DEST}.raw.bch"
        set bch [open $bchfn "wb"]

        puts "Writing BCH to $bchfn..."
        puts [format "0x%08x - 0x%08x : $bchfn (bch of $SOURCE)" $addr_bch [expr {$addr_data + $SIZEB - 1}]]
        foreach b [lreverse $bchs] {
                w8 $dch $b
        }
        close $dch


        set elffn "${DEST}.elf"
        exec $OBJCOPY \
                -I binary \
                -O elf32-sparc \
                -B sparc \
                --change-section-address .data=$addr_data \
                --add-section .bch=$bchfn \
                --set-section-flags .bch=contents,alloc,load,data \
                --change-section-address .bch=$addr_bch \
                $datafn \
                $elffn
}

if {$opt_elf} {
        output_elf
} else {
        output_bin
}

