#include <adchelper.h>
#include <bcc/bsp_pnp.h>
#include <gr716/adc0_regs.h>
#include <drv/osal.h>

static struct adc0_regs *UNIT_TO_REGS(
        int unit
)
{
        const uintptr_t REG_ADDR = GAISLER_ADC_0_PNP_APB + unit * 0x00001000;
        struct adc0_regs *const regs = (struct adc0_regs *) REG_ADDR;
        return regs;
}

/* unit: 0..7 */
int adc_config(int unit, const struct adc_config *cfg)
{
        volatile struct adc0_regs *const regs = UNIT_TO_REGS(unit);
        unsigned int freq;
        unsigned int scaler = 0;
        unsigned int divisor = 0;

        freq = osal_busfreq();
        /* try to prevent division by zero when not used */
        if (cfg->rate) {
                scaler = (freq / (cfg->rate * 20));
        }
        if (scaler) {
                divisor = (cfg->seqrate + 1) / scaler;
        }

        adc_disable(unit);

        /* pre-amplifier */
        {
                uint32_t pacfg;

                pacfg = ADC0_PACFG_AB;
                if (cfg->gain != ADC_GAIN_BYPASS) {
                        pacfg = cfg->gain & ADC0_PACFG_AG;
                }
                regs->pacfg = pacfg;
        }

        /* adc config */
        {
                uint32_t acfg = 0;

                if (cfg->inputmode == ADC_INMODE_SINGLE) {
                        acfg |= ADC0_ACFG_AM;
                }
                acfg |= (cfg->channel << ADC0_ACFG_AI_BIT) & ADC0_ACFG_AI;
                acfg |= (scaler << ADC0_ACFG_AC_BIT) & ADC0_ACFG_AC;
                regs->acfg = acfg;
        }

        /* sampling control */
        {
                uint32_t asampc;
                const int ASAMPC_AO = 0xffff;

                asampc  = cfg->oversampling & ASAMPC_AO;
                asampc |= (cfg->oversampling_events << ADC0_ASAMPC_AE_BIT) &
                    ADC0_ASAMPC_AE;
                regs->asampc = asampc;
        }

        /* sequencer */
        {
                uint32_t aseqc = 0;

                if (cfg->mode == ADC_MODE_SEQ) {
                        aseqc |= ADC0_ASEQC_SE;
                } else if (cfg->mode == ADC_MODE_SEQ_CONT) {
                        aseqc |= ADC0_ASEQC_SE;
                        aseqc |= ADC0_ASEQC_SC;
                }

                if (cfg->sync) {
                        /* qualifies ASYNC */
                        aseqc |= ADC0_ASEQC_SQ;
                }

                /* sequence divisor */
                aseqc |= divisor & ADC0_ASEQC_SD;

                regs->aseqc = aseqc;
        }

        regs->async = cfg->sync_source & ADC0_ASYNC_SYNC;

        return 0;
}

int adc_enable(int unit)
{
        volatile struct adc0_regs *const regs = UNIT_TO_REGS(unit);

        regs->acfg |= ADC0_ACFG_AE;

        return 0;
}

int adc_disable(int unit)
{
        volatile struct adc0_regs *const regs = UNIT_TO_REGS(unit);

        regs->acfg &= ~ADC0_ACFG_AE;

        return 0;
}

int adc_start(int unit)
{
        volatile struct adc0_regs *const regs = UNIT_TO_REGS(unit);

        regs->acfg |= ADC0_ACFG_AS;

        return 0;
}

int adc_stop(int unit)
{
        volatile struct adc0_regs *const regs = UNIT_TO_REGS(unit);

        regs->acfg &= ~ADC0_ACFG_AS;

        return 0;
}

unsigned int adc_get(int unit)
{
        volatile struct adc0_regs *const regs = UNIT_TO_REGS(unit);

        return regs->asq0 & ADC0_ASQ0_AD;
}

int adc_intmask(int unit, unsigned int mask)
{
        volatile struct adc0_regs *const regs = UNIT_TO_REGS(unit);
        regs->amask = mask & 0xf;
        return 0;
}

unsigned int adc_getint(int unit, unsigned int clearmask)
{
        volatile struct adc0_regs *const regs = UNIT_TO_REGS(unit);
        unsigned int oldval;
        oldval = regs->aint;
        regs->aint = clearmask & 0xf;
        return oldval;
}

