#ifndef ADC0_REGS_H
#define ADC0_REGS_H

/**
 * @file
 *
 * @brief Register description for adc0
 *
 */

/* version: GRMON 3.0.15 */

#include <stdint.h>

/**
 * @brief GAISLER_UNKNOWN_0
 *
 * address: 0x80400000
 *
 * Offset | Name       | Description
 * ------ | ---------- | --------------------------------
 * 0x0000 | acfg       | ADC Control
 * 0x0004 | asampc     | ADC Sampling Control
 * 0x0008 | aseqc      | ADC Sequence Control
 * 0x000c | async      | ADC Sequence Synchronization
 * 0x0010 | asts       | ADC status
 * 0x0014 | aint       | Interrupt
 * 0x0018 | amask      | Interrupt Mask
 * 0x001c | pacfg      | Pre-Amplifier Control
 * 0x0020 | aht        | High Level detection
 * 0x0024 | alt        | Low Level detection
 * 0x002c | asq0       | ADC Sequence status 0
 * 0x0030 | asq1       | ADC Sequence status 1
 * 0x0034 | asq2       | ADC Sequence status 2
 * 0x0038 | asq3       | ADC Sequence status 3
 */
struct adc0_regs {
  /**
   * @brief ADC Control
   *
   * offset: 0x0000
   *
   * Bit    | Name       | Description
   * ------ | ---------- | --------------------------------
   * 31:16  | ac         | Scaler reload bits
   * 15     | ae         | Enable On-chip ADC
   * 14     | as         | Start conversion
   *  8     | ah         | High Range check enable
   *  7     | al         | Low Range check enable
   *  5:2   | ai         | Channel select
   *  0     | am         | Select single ended mode
   */
  uint32_t acfg;

  /**
   * @brief ADC Sampling Control
   *
   * offset: 0x0004
   *
   * Bit    | Name       | Description
   * ------ | ---------- | --------------------------------
   * 17:16  | ae         | ADC Events
   * 15:0   | a0         | ADC Oversampling
   */
  uint32_t asampc;

  /**
   * @brief ADC Sequence Control
   *
   * offset: 0x0008
   *
   * Bit    | Name       | Description
   * ------ | ---------- | --------------------------------
   * 31     | sq         | Sequence synchronization enable
   * 29     | se         | Sequence enable
   * 28     | sc         | Sequence continuously enabled
   * 27     | ac         | Auto clear interrupt
   * 15:0   | sd         | Sequence divisor
   */
  uint32_t aseqc;

  /**
   * @brief ADC Sequence Synchronization
   *
   * offset: 0x000c
   *
   * Bit    | Name       | Description
   * ------ | ---------- | --------------------------------
   *  5:0   | sync       | Synchronize source select
   */
  uint32_t async;

  /**
   * @brief ADC status
   *
   * offset: 0x0010
   *
   * Bit    | Name       | Description
   * ------ | ---------- | --------------------------------
   * 31     | ae         | End of conversion
   * 19:0   | ad         | Digital sampled value
   */
  uint32_t asts;

  /**
   * @brief Interrupt
   *
   * offset: 0x0014
   *
   * Bit    | Name       | Description
   * ------ | ---------- | --------------------------------
   *  3     | is         | End of Sequence
   *  2     | ih         | High level detection
   *  1     | il         | Low level detection
   *  0     | ie         | End of conversion
   */
  uint32_t aint;

  /**
   * @brief Interrupt Mask
   *
   * offset: 0x0018
   *
   * Bit    | Name       | Description
   * ------ | ---------- | --------------------------------
   *  3     | ms         | End of Sequence
   *  2     | mh         | High level detection
   *  1     | ml         | Low level detection
   *  0     | me         | End of conversion
   */
  uint32_t amask;

  /**
   * @brief Pre-Amplifier Control
   *
   * offset: 0x001c
   *
   * Bit    | Name       | Description
   * ------ | ---------- | --------------------------------
   *  4     | al         | Amplifier Autozeroing low
   *  3     | ah         | Amplifier Autozerioing high
   *  2     | ab         | Amplifier Bypass
   *  1:0   | ag         | Amplifier Gain
   */
  uint32_t pacfg;

  /**
   * @brief High Level detection
   *
   * offset: 0x0020
   *
   * Bit    | Name       | Description
   * ------ | ---------- | --------------------------------
   * 19:0   | aht        | High Level detection threshold
   */
  uint32_t aht;

  /**
   * @brief Low Level detection
   *
   * offset: 0x0024
   *
   * Bit    | Name       | Description
   * ------ | ---------- | --------------------------------
   * 19:0   | alt        | Low Level detection threshold
   */
  uint32_t alt;

  uint32_t _pad0x0028[0x0004 / 4];
  /**
   * @brief ADC Sequence status 0
   *
   * offset: 0x002c
   *
   * Bit    | Name       | Description
   * ------ | ---------- | --------------------------------
   * 19:0   | ad         | Digital sampled value 0
   */
  uint32_t asq0;

  /**
   * @brief ADC Sequence status 1
   *
   * offset: 0x0030
   *
   * Bit    | Name       | Description
   * ------ | ---------- | --------------------------------
   * 19:0   | ad         | Digital sampled value 1
   */
  uint32_t asq1;

  /**
   * @brief ADC Sequence status 2
   *
   * offset: 0x0034
   *
   * Bit    | Name       | Description
   * ------ | ---------- | --------------------------------
   * 19:0   | ad         | Digital sampled value 2
   */
  uint32_t asq2;

  /**
   * @brief ADC Sequence status 3
   *
   * offset: 0x0038
   *
   * Bit    | Name       | Description
   * ------ | ---------- | --------------------------------
   * 19:0   | ad         | Digital sampled value 3
   */
  uint32_t asq3;

};

/** @brief Scaler reload bits bit number*/
#define ADC0_ACFG_AC_BIT                         16
/** @brief Enable On-chip ADC bit number*/
#define ADC0_ACFG_AE_BIT                         15
/** @brief Start conversion bit number*/
#define ADC0_ACFG_AS_BIT                         14
/** @brief High Range check enable bit number*/
#define ADC0_ACFG_AH_BIT                          8
/** @brief Low Range check enable bit number*/
#define ADC0_ACFG_AL_BIT                          7
/** @brief Channel select bit number*/
#define ADC0_ACFG_AI_BIT                          2
/** @brief Select single ended mode bit number*/
#define ADC0_ACFG_AM_BIT                          0

/** @brief ADC Events bit number*/
#define ADC0_ASAMPC_AE_BIT                       16
/** @brief ADC Oversampling bit number*/
#define ADC0_ASAMPC_A0_BIT                        0

/** @brief Sequence synchronization enable bit number*/
#define ADC0_ASEQC_SQ_BIT                        31
/** @brief Sequence enable bit number*/
#define ADC0_ASEQC_SE_BIT                        29
/** @brief Sequence continuously enabled bit number*/
#define ADC0_ASEQC_SC_BIT                        28
/** @brief Auto clear interrupt bit number*/
#define ADC0_ASEQC_AC_BIT                        27
/** @brief Sequence divisor bit number*/
#define ADC0_ASEQC_SD_BIT                         0

/** @brief Synchronize source select bit number*/
#define ADC0_ASYNC_SYNC_BIT                       0

/** @brief End of conversion bit number*/
#define ADC0_ASTS_AE_BIT                         31
/** @brief Digital sampled value bit number*/
#define ADC0_ASTS_AD_BIT                          0

/** @brief End of Sequence bit number*/
#define ADC0_AINT_IS_BIT                          3
/** @brief High level detection bit number*/
#define ADC0_AINT_IH_BIT                          2
/** @brief Low level detection bit number*/
#define ADC0_AINT_IL_BIT                          1
/** @brief End of conversion bit number*/
#define ADC0_AINT_IE_BIT                          0

/** @brief End of Sequence bit number*/
#define ADC0_AMASK_MS_BIT                         3
/** @brief High level detection bit number*/
#define ADC0_AMASK_MH_BIT                         2
/** @brief Low level detection bit number*/
#define ADC0_AMASK_ML_BIT                         1
/** @brief End of conversion bit number*/
#define ADC0_AMASK_ME_BIT                         0

/** @brief Amplifier Autozeroing low bit number*/
#define ADC0_PACFG_AL_BIT                         4
/** @brief Amplifier Autozerioing high bit number*/
#define ADC0_PACFG_AH_BIT                         3
/** @brief Amplifier Bypass bit number*/
#define ADC0_PACFG_AB_BIT                         2
/** @brief Amplifier Gain bit number*/
#define ADC0_PACFG_AG_BIT                         0

/** @brief High Level detection threshold bit number*/
#define ADC0_AHT_AHT_BIT                          0

/** @brief Low Level detection threshold bit number*/
#define ADC0_ALT_ALT_BIT                          0

/** @brief Digital sampled value 0 bit number*/
#define ADC0_ASQ0_AD_BIT                          0

/** @brief Digital sampled value 1 bit number*/
#define ADC0_ASQ1_AD_BIT                          0

/** @brief Digital sampled value 2 bit number*/
#define ADC0_ASQ2_AD_BIT                          0

/** @brief Digital sampled value 3 bit number*/
#define ADC0_ASQ3_AD_BIT                          0

/** @brief Scaler reload bits mask */
#define ADC0_ACFG_AC                             0xffff0000
/** @brief Enable On-chip ADC mask */
#define ADC0_ACFG_AE                             0x00008000
/** @brief Start conversion mask */
#define ADC0_ACFG_AS                             0x00004000
/** @brief High Range check enable mask */
#define ADC0_ACFG_AH                             0x00000100
/** @brief Low Range check enable mask */
#define ADC0_ACFG_AL                             0x00000080
/** @brief Channel select mask */
#define ADC0_ACFG_AI                             0x0000003c
/** @brief Select single ended mode mask */
#define ADC0_ACFG_AM                             0x00000001

/** @brief ADC Events mask */
#define ADC0_ASAMPC_AE                           0x00030000
/** @brief ADC Oversampling mask */
#define ADC0_ASAMPC_A0                           0x0000ffff

/** @brief Sequence synchronization enable mask */
#define ADC0_ASEQC_SQ                            0x80000000
/** @brief Sequence enable mask */
#define ADC0_ASEQC_SE                            0x20000000
/** @brief Sequence continuously enabled mask */
#define ADC0_ASEQC_SC                            0x10000000
/** @brief Auto clear interrupt mask */
#define ADC0_ASEQC_AC                            0x08000000
/** @brief Sequence divisor mask */
#define ADC0_ASEQC_SD                            0x0000ffff

/** @brief Synchronize source select mask */
#define ADC0_ASYNC_SYNC                          0x0000003f

/** @brief End of conversion mask */
#define ADC0_ASTS_AE                             0x80000000
/** @brief Digital sampled value mask */
#define ADC0_ASTS_AD                             0x000fffff

/** @brief End of Sequence mask */
#define ADC0_AINT_IS                             0x00000008
/** @brief High level detection mask */
#define ADC0_AINT_IH                             0x00000004
/** @brief Low level detection mask */
#define ADC0_AINT_IL                             0x00000002
/** @brief End of conversion mask */
#define ADC0_AINT_IE                             0x00000001

/** @brief End of Sequence mask */
#define ADC0_AMASK_MS                            0x00000008
/** @brief High level detection mask */
#define ADC0_AMASK_MH                            0x00000004
/** @brief Low level detection mask */
#define ADC0_AMASK_ML                            0x00000002
/** @brief End of conversion mask */
#define ADC0_AMASK_ME                            0x00000001

/** @brief Amplifier Autozeroing low mask */
#define ADC0_PACFG_AL                            0x00000010
/** @brief Amplifier Autozerioing high mask */
#define ADC0_PACFG_AH                            0x00000008
/** @brief Amplifier Bypass mask */
#define ADC0_PACFG_AB                            0x00000004
/** @brief Amplifier Gain mask */
#define ADC0_PACFG_AG                            0x00000003

/** @brief High Level detection threshold mask */
#define ADC0_AHT_AHT                             0x000fffff

/** @brief Low Level detection threshold mask */
#define ADC0_ALT_ALT                             0x000fffff

/** @brief Digital sampled value 0 mask */
#define ADC0_ASQ0_AD                             0x000fffff

/** @brief Digital sampled value 1 mask */
#define ADC0_ASQ1_AD                             0x000fffff

/** @brief Digital sampled value 2 mask */
#define ADC0_ASQ2_AD                             0x000fffff

/** @brief Digital sampled value 3 mask */
#define ADC0_ASQ3_AD                             0x000fffff


#endif

