#ifndef MCTRL0_REGS_H
#define MCTRL0_REGS_H

/**
 * @file
 *
 * @brief Register description for mctrl0
 *
 */

/* version: GRMON 3.0.15 */

#include <stdint.h>

/**
 * @brief GAISLER_FTMCTRL_0
 *
 * address: 0x80000000
 *
 * Offset | Name       | Description
 * ------ | ---------- | --------------------------------
 * 0x0000 | mcfg1      | Memory config register 1
 * 0x0004 | mcfg2      | Memory config register 2
 * 0x0008 | mcfg3      | Memory config register 3
 * 0x0010 | mcfg5      | Memory config register 5
 * 0x0014 | mcfg6      | Memory config register 6
 */
struct mctrl0_regs {
  /**
   * @brief Memory config register 1
   *
   * offset: 0x0000
   *
   * Bit    | Name       | Description
   * ------ | ---------- | --------------------------------
   * 30     | pbrdy      | PROM area bus ready enable
   * 29     | abrdy      | Asynchronous bus ready enable
   * 28:27  | iobusw     | I/O bus width
   * 26     | ibrdy      | I/O bus ready enable
   * 25     | bexcn      | Bus error enable
   * 23:20  | iows       | I/O wait states
   * 19     | ioen       | I/O enable
   * 17:14  | prombanksz | PROM bank size
   * 11     | pwen       | PROM write enable
   *  9:8   | promwidth  | PROM width
   *  7:4   | promwws    | PROM write wait states
   *  3:0   | promrws    | PROM read wait states
   */
  uint32_t mcfg1;

  /**
   * @brief Memory config register 2
   *
   * offset: 0x0004
   *
   * Bit    | Name       | Description
   * ------ | ---------- | --------------------------------
   * 31     | sdramrf    | SDRAM refresh enable
   * 30     | sdramtrp   | SDRAM TRP parameter
   * 29:27  | sdramtrfc  | SDRAM TRFC parameter
   * 26     | sdramtcas  | SDRAM TCAS parameter
   * 25:23  | sdrambanksz | SDRAM bank size
   * 22:21  | sdramcolsz | SDRAM column size
   * 20:19  | sdramcmd   | SDRAM command
   * 18     | d64        | SDRAM 64-bit data bus
   * 17     | sdpb       | SDRAM page burst
   * 14     | se         | SDRAM enable
   * 13     | si         | SRAM disable
   * 12:9   | rambanksz  | RAM bank size
   *  7     | rbrdy      | RAM bus read enable
   *  6     | rmw        | Read-modify-write enable
   *  5:4   | ramwidth   | RAM width
   *  3:2   | ramwws     | RAM write wait states
   *  1:0   | ramrws     | RAM read wait states
   */
  uint32_t mcfg2;

  /**
   * @brief Memory config register 3
   *
   * offset: 0x0008
   *
   * Bit    | Name       | Description
   * ------ | ---------- | --------------------------------
   * 28     | rse        | Reed-Solomon EDAC enable
   * 27     | me         | Memory EDAC available
   * 26:12  | sdramreload | SDRAM refresh counter reload value
   * 11     | wb         | EDAC diagnostic write bypass enable
   * 10     | rb         | EDAC diagnostic read bypass enable
   *  9     | re         | RAM EDAC enable
   *  8     | pe         | PROM EDAC enable
   *  7:0   | tcb        | Test checkbits
   */
  uint32_t mcfg3;

  uint32_t _pad0x000c[0x0004 / 4];
  /**
   * @brief Memory config register 5
   *
   * offset: 0x0010
   *
   * Bit    | Name       | Description
   * ------ | ---------- | --------------------------------
   * 29:23  | iohws      | IO lead out
   * 13:7   | romhws     | ROM lead out
   */
  uint32_t mcfg5;

  /**
   * @brief Memory config register 6
   *
   * offset: 0x0014
   *
   * Bit    | Name       | Description
   * ------ | ---------- | --------------------------------
   * 13:7   | ramhws     | RAM lead out
   */
  uint32_t mcfg6;

};

/** @brief PROM area bus ready enable bit number*/
#define MCTRL0_MCFG1_PBRDY_BIT                   30
/** @brief Asynchronous bus ready enable bit number*/
#define MCTRL0_MCFG1_ABRDY_BIT                   29
/** @brief I/O bus width bit number*/
#define MCTRL0_MCFG1_IOBUSW_BIT                  27
/** @brief I/O bus ready enable bit number*/
#define MCTRL0_MCFG1_IBRDY_BIT                   26
/** @brief Bus error enable bit number*/
#define MCTRL0_MCFG1_BEXCN_BIT                   25
/** @brief I/O wait states bit number*/
#define MCTRL0_MCFG1_IOWS_BIT                    20
/** @brief I/O enable bit number*/
#define MCTRL0_MCFG1_IOEN_BIT                    19
/** @brief PROM bank size bit number*/
#define MCTRL0_MCFG1_PROMBANKSZ_BIT              14
/** @brief PROM write enable bit number*/
#define MCTRL0_MCFG1_PWEN_BIT                    11
/** @brief PROM width bit number*/
#define MCTRL0_MCFG1_PROMWIDTH_BIT                8
/** @brief PROM write wait states bit number*/
#define MCTRL0_MCFG1_PROMWWS_BIT                  4
/** @brief PROM read wait states bit number*/
#define MCTRL0_MCFG1_PROMRWS_BIT                  0

/** @brief SDRAM refresh enable bit number*/
#define MCTRL0_MCFG2_SDRAMRF_BIT                 31
/** @brief SDRAM TRP parameter bit number*/
#define MCTRL0_MCFG2_SDRAMTRP_BIT                30
/** @brief SDRAM TRFC parameter bit number*/
#define MCTRL0_MCFG2_SDRAMTRFC_BIT               27
/** @brief SDRAM TCAS parameter bit number*/
#define MCTRL0_MCFG2_SDRAMTCAS_BIT               26
/** @brief SDRAM bank size bit number*/
#define MCTRL0_MCFG2_SDRAMBANKSZ_BIT             23
/** @brief SDRAM column size bit number*/
#define MCTRL0_MCFG2_SDRAMCOLSZ_BIT              21
/** @brief SDRAM command bit number*/
#define MCTRL0_MCFG2_SDRAMCMD_BIT                19
/** @brief SDRAM 64-bit data bus bit number*/
#define MCTRL0_MCFG2_D64_BIT                     18
/** @brief SDRAM page burst bit number*/
#define MCTRL0_MCFG2_SDPB_BIT                    17
/** @brief SDRAM enable bit number*/
#define MCTRL0_MCFG2_SE_BIT                      14
/** @brief SRAM disable bit number*/
#define MCTRL0_MCFG2_SI_BIT                      13
/** @brief RAM bank size bit number*/
#define MCTRL0_MCFG2_RAMBANKSZ_BIT                9
/** @brief RAM bus read enable bit number*/
#define MCTRL0_MCFG2_RBRDY_BIT                    7
/** @brief Read-modify-write enable bit number*/
#define MCTRL0_MCFG2_RMW_BIT                      6
/** @brief RAM width bit number*/
#define MCTRL0_MCFG2_RAMWIDTH_BIT                 4
/** @brief RAM write wait states bit number*/
#define MCTRL0_MCFG2_RAMWWS_BIT                   2
/** @brief RAM read wait states bit number*/
#define MCTRL0_MCFG2_RAMRWS_BIT                   0

/** @brief Reed-Solomon EDAC enable bit number*/
#define MCTRL0_MCFG3_RSE_BIT                     28
/** @brief Memory EDAC available bit number*/
#define MCTRL0_MCFG3_ME_BIT                      27
/** @brief SDRAM refresh counter reload value bit number*/
#define MCTRL0_MCFG3_SDRAMRELOAD_BIT             12
/** @brief EDAC diagnostic write bypass enable bit number*/
#define MCTRL0_MCFG3_WB_BIT                      11
/** @brief EDAC diagnostic read bypass enable bit number*/
#define MCTRL0_MCFG3_RB_BIT                      10
/** @brief RAM EDAC enable bit number*/
#define MCTRL0_MCFG3_RE_BIT                       9
/** @brief PROM EDAC enable bit number*/
#define MCTRL0_MCFG3_PE_BIT                       8
/** @brief Test checkbits bit number*/
#define MCTRL0_MCFG3_TCB_BIT                      0

/** @brief IO lead out bit number*/
#define MCTRL0_MCFG5_IOHWS_BIT                   23
/** @brief ROM lead out bit number*/
#define MCTRL0_MCFG5_ROMHWS_BIT                   7

/** @brief RAM lead out bit number*/
#define MCTRL0_MCFG6_RAMHWS_BIT                   7

/** @brief PROM area bus ready enable mask */
#define MCTRL0_MCFG1_PBRDY                       0x40000000
/** @brief Asynchronous bus ready enable mask */
#define MCTRL0_MCFG1_ABRDY                       0x20000000
/** @brief I/O bus width mask */
#define MCTRL0_MCFG1_IOBUSW                      0x18000000
/** @brief I/O bus ready enable mask */
#define MCTRL0_MCFG1_IBRDY                       0x04000000
/** @brief Bus error enable mask */
#define MCTRL0_MCFG1_BEXCN                       0x02000000
/** @brief I/O wait states mask */
#define MCTRL0_MCFG1_IOWS                        0x00f00000
/** @brief I/O enable mask */
#define MCTRL0_MCFG1_IOEN                        0x00080000
/** @brief PROM bank size mask */
#define MCTRL0_MCFG1_PROMBANKSZ                  0x0003c000
/** @brief PROM write enable mask */
#define MCTRL0_MCFG1_PWEN                        0x00000800
/** @brief PROM width mask */
#define MCTRL0_MCFG1_PROMWIDTH                   0x00000300
/** @brief PROM write wait states mask */
#define MCTRL0_MCFG1_PROMWWS                     0x000000f0
/** @brief PROM read wait states mask */
#define MCTRL0_MCFG1_PROMRWS                     0x0000000f

/** @brief SDRAM refresh enable mask */
#define MCTRL0_MCFG2_SDRAMRF                     0x80000000
/** @brief SDRAM TRP parameter mask */
#define MCTRL0_MCFG2_SDRAMTRP                    0x40000000
/** @brief SDRAM TRFC parameter mask */
#define MCTRL0_MCFG2_SDRAMTRFC                   0x38000000
/** @brief SDRAM TCAS parameter mask */
#define MCTRL0_MCFG2_SDRAMTCAS                   0x04000000
/** @brief SDRAM bank size mask */
#define MCTRL0_MCFG2_SDRAMBANKSZ                 0x03800000
/** @brief SDRAM column size mask */
#define MCTRL0_MCFG2_SDRAMCOLSZ                  0x00600000
/** @brief SDRAM command mask */
#define MCTRL0_MCFG2_SDRAMCMD                    0x00180000
/** @brief SDRAM 64-bit data bus mask */
#define MCTRL0_MCFG2_D64                         0x00040000
/** @brief SDRAM page burst mask */
#define MCTRL0_MCFG2_SDPB                        0x00020000
/** @brief SDRAM enable mask */
#define MCTRL0_MCFG2_SE                          0x00004000
/** @brief SRAM disable mask */
#define MCTRL0_MCFG2_SI                          0x00002000
/** @brief RAM bank size mask */
#define MCTRL0_MCFG2_RAMBANKSZ                   0x00001e00
/** @brief RAM bus read enable mask */
#define MCTRL0_MCFG2_RBRDY                       0x00000080
/** @brief Read-modify-write enable mask */
#define MCTRL0_MCFG2_RMW                         0x00000040
/** @brief RAM width mask */
#define MCTRL0_MCFG2_RAMWIDTH                    0x00000030
/** @brief RAM write wait states mask */
#define MCTRL0_MCFG2_RAMWWS                      0x0000000c
/** @brief RAM read wait states mask */
#define MCTRL0_MCFG2_RAMRWS                      0x00000003

/** @brief Reed-Solomon EDAC enable mask */
#define MCTRL0_MCFG3_RSE                         0x10000000
/** @brief Memory EDAC available mask */
#define MCTRL0_MCFG3_ME                          0x08000000
/** @brief SDRAM refresh counter reload value mask */
#define MCTRL0_MCFG3_SDRAMRELOAD                 0x07fff000
/** @brief EDAC diagnostic write bypass enable mask */
#define MCTRL0_MCFG3_WB                          0x00000800
/** @brief EDAC diagnostic read bypass enable mask */
#define MCTRL0_MCFG3_RB                          0x00000400
/** @brief RAM EDAC enable mask */
#define MCTRL0_MCFG3_RE                          0x00000200
/** @brief PROM EDAC enable mask */
#define MCTRL0_MCFG3_PE                          0x00000100
/** @brief Test checkbits mask */
#define MCTRL0_MCFG3_TCB                         0x000000ff

/** @brief IO lead out mask */
#define MCTRL0_MCFG5_IOHWS                       0x3f800000
/** @brief ROM lead out mask */
#define MCTRL0_MCFG5_ROMHWS                      0x00003f80

/** @brief RAM lead out mask */
#define MCTRL0_MCFG6_RAMHWS                      0x00003f80


#endif

