#!/usr/bin/env tclsh

# Tool for transforming an ELF file into an ASW image. Uses GNU binutils.  By
# default a binary file ASW image file is created. If the -a option is present,
# then an ELF file suitable for loading with GRMON2 is also created. This ELF
# image contains information on where (ASM_ADDR) to load the image data.
set util_rev 1.1
set usage "Usage: $argv0 \
        \[-a ASM_ADDR\] \
        \[-o OUTFILE\] \
        INFILE..."

source [file join [file dirname [info script]] "crc16.tcl"]

set OBJDUMP sparc-gaisler-elf-objdump
set OBJCOPY sparc-gaisler-elf-objcopy
set NUM_SECTIONS 8
set SECTION_WORDS 5
# b,a   20000ac
# 2000000:       30 80 00 2b     b,a   20000ac <blupp>
set id 0x3080002b
set ASW_SECTION_FLAGS_EN 1
set ASW_SECTION_FLAGS_IGNORE_CRC 2

# From filename, generate objdump lines.
proc gen_lines {fn} {
        global OBJDUMP

        # Collect information on LOAD sections from objdump.
        set res [exec $OBJDUMP -h $fn | grep -B 1 LOAD | grep -v - -- ]

        # Format objdump output to convenient lines.
        set sres [split $res "\n"]
        foreach {a b} $sres {
                set line "$a $b"
                set line [regexp -all -inline {\S+} $line]
                lappend lines $line
        }
        return $lines
}

# From objdump lines and file name, generate a dict with sections
proc collect {lines fn} {
        global HEADER_BYTES
        global sourcebyte

        # Collect the information in a dictionary.
        # Byte offset incrementor
        puts "  $fn"
        foreach line $lines {
                set secname "[lindex $line 1]"
                set name "${fn}_${secname}"
                set src [expr {$sourcebyte/4}]
                set size "0x[lindex $line 2]"
                set length [expr {$size/4}]
                set lma "0x[lindex $line 4]"
                dict set secs $name fn $fn
                dict set secs $name secname $secname
                dict set secs $name source $src
                dict set secs $name dest $lma
                dict set secs $name length $length
                dict set secs $name sfn "${name}.bin"
                puts -nonewline "    $secname\tsource=[format "%08x" $src]"
                puts "  dest=[format "%08x" $lma]  length=[format "%08x" $length]"
                incr sourcebyte $size
        }
        return $secs
}

if {0 == $argc} {
        puts $usage
        exit 1
}

# Parse optional arguments
set state flag
foreach arg [lrange $argv 0 end] {
        switch -- $state {
                flag {
                        switch -glob -- $arg {
                                -n* {set state n}
                                -a* {set state a}
                                -o* {set state o}
                                -* {puts $usage; exit 1}
                                default {lappend fns $arg}
                        }
                }
                n {
                        # NOTE: GR716 ROM boot loader assmes 8
                        set NUM_SECTIONS $arg
                        set state flag
                }
                a {
                        set ASM_ADDR $arg
                        set state flag
                }
                o {
                        set ofn $arg
                        set state flag
                }
        }
}
unset state

# OUTFILE defaults to first INFILE
set fn0 [lindex $fns 0]
if {![info exists ofn]} {
        set ofn $fn0
}

set HEADER_BYTES [expr {4 + 4 + $NUM_SECTIONS*$SECTION_WORDS*4 + 2 + 2}]

puts "  NUM_SECTIONS: $NUM_SECTIONS"
puts "     INFILE(S): $fns"
puts "       OUTFILE: $ofn"

# Get entry point, comes from the LAST file on command line
set fnep [lindex $fns end]
set ep [exec $OBJDUMP -f $fnep | grep "start address"]
set ep [lindex $ep 2]

puts "Collecting LOAD sections..."
set sourcebyte ${HEADER_BYTES}
set secs [dict create]
foreach fn $fns {
        set lines [gen_lines $fn]
        set moresecs [collect $lines $fn]
        set secs [dict merge $secs $moresecs]
        unset lines
        unset moresecs
}
unset fn

# Print some information on what was collected.
set count [dict size $secs]
puts "Collected $count LOAD sections"
puts "Entry point: $ep ($fnep)"
#dict for {k v} $secs {
#        puts "  Section $k: $v"
#}

# Fail if to many LOAD sections in file.
if {$NUM_SECTIONS < $count} {
        puts "$argv0: too many LOAD sections in '$fns'"
        exit 1
}

# Generate one binary file per section.
puts "Dumping LOAD section data to binary files:"
dict for {k v} $secs {
        set secname [dict get $v secname]
        set thisfn [dict get $v fn]
        set sfn [dict get $v sfn]
        puts "  $sfn"
        exec $OBJCOPY $thisfn $sfn -j $secname -O binary
}

# Generate output binary

set hdrfn "$ofn.hdr"
puts "Generating header $hdrfn..."
set ch [open $hdrfn "w+b"]
w32 $ch $id
w32 $ch $ep
set sindex 0

# Disable section with index 0 and install ep in its dest field. The GR716 boot
# ROM assumes that the application entry point is located in image memory
# location 0x10. This correspond with the "dest" field of section 0.
set section0_is_ep true
if {$section0_is_ep} {
        w32 $ch 0               ;# asw->section[0].flags
        w32 $ch 0x45502d3e      ;# asw->section[0].source
        w32 $ch $ep             ;# asw->section[0].dest
        w32 $ch 0x20475237      ;# asw->section[0].length
        w32 $ch 0x31362000      ;# asw->section[0].{datacksum,rsv0}
        incr sindex
}

dict for {k v} $secs {
        w32 $ch $ASW_SECTION_FLAGS_EN
        w32 $ch [dict get $v source]
        w32 $ch [dict get $v dest]
        w32 $ch [dict get $v length]
        # Calculate section file checksum.
        set sfn [dict get $v sfn]
        puts "  $sfn"
        set datacksum [filecksum $sfn]
        w16 $ch $datacksum
        # rsv0
        w16 $ch 0
        incr sindex
}

# Set disabled sections
while {$sindex < $NUM_SECTIONS} {
        for {set i 0} {$i < $SECTION_WORDS} {incr i} {
                w32 $ch 0
        }
        incr sindex
}

# rsv0
w16 $ch 0
# Calculate CRC16 on ($HEADER_BYTES-2) bytes of the output file.
seek $ch 0
set cksum [chcksum $ch]
w16 $ch $cksum

close $ch

set imgfn "$ofn"
puts "Concatenating $hdrfn and LOAD sections to $imgfn"

exec cat $hdrfn > $imgfn
exec rm $hdrfn
dict for {k v} $secs {
        set sfn [dict get $v sfn]
        exec cat $sfn >> $imgfn
        exec rm $sfn
}

# Only generate ELF if used supplied ASM_ADDR.
if {[info exists ASM_ADDR]} {
        set elffn "$imgfn.elf"
        puts "Generating $elffn image at $ASM_ADDR for $imgfn"
        exec $OBJCOPY -I binary -O elf32-sparc -B sparc --change-addresses $ASM_ADDR $imgfn $elffn
}

