/*
 * ABOUT
 *   This is a memory test program which can be used to detect problems with
 *   external memory connection, for example due to signalling problems.
 *
 * CONFIGURE
 *   Set the parameters TADDR and TSIZE in the source.
 *
 * BUILD EXAMPLE
 *   sparc-gaisler-elf-gcc -std=c99 -Wall -Wextra -pedantic -Os memtest.c -o memtest.elf
 *
 *   The compiler can be replaced with sparc-elf-gcc, sparc-rtems-gcc, etc.
 *
 * LOAD AND RUN
 *   Load with GRMON on non-GR716 systems:
 *   grmon3> load memtest.elf
 *   grmon3> stack 0x40100000
 *   grmon3> dcache disable
 *   grmon3> run
 *
 *   Load with GRMON on GR716 systems:
 *   grmon3> grcg enable 16
 *   grmon3> forward enable uart0
 *   grmon3> load memtest.elf
 *   grmon3> run
 *
 * NOTE
 *   - It is important that the stack is set such that it does not collide with
 *   the tested memory. The stack extendes downwards by the CPU.
 *   - It is important that the TADDR is set such that it does not collide with
 *   the test program itself. The example above should work on a system with 16
 *   MiB RAM.
 *
 * OUTPUT
 *   The program tests memory in a loop and prints a dot (".") each time the
 *   full test region defined by CFG_TADDR,CFG_TSIZE has been tested.
 *
 * Written by Martin Aberg, Cobham Gaisler AB
 */

#include <stdio.h>

#ifndef CFG_TRAP_ON_MISMATCH
 #define CFG_TRAP_ON_MISMATCH 0
#endif

#ifndef CFG_TADDR
 #define CFG_TADDR 0x40000000
#endif

#ifndef CFG_TSIZE
 #define CFG_TSIZE (2 * 1024 * 1024)
#endif

#ifndef CFG_TESTNUM
 #define CFG_TESTNUM 0
#endif

/* TADDR: Start address for memory to test */
/* TSIZE: Number of bytes to test */
uint32_t *const TADDR = (void *) CFG_TADDR;
const size_t TSIZE    =          CFG_TSIZE;

uint32_t *memtest_seqdata(volatile uint32_t *addr, size_t size);
uint32_t *memtest_simple_a(volatile uint32_t *addr, size_t size, int forever);

/*
 * Call this function to abort execution and return to GRMON. After the trap,
 * inspect system state in GRMON with:
 *   grmon3> bt
 *   grmon3> dcache diag
 *   grmon3> hist 2048
 *   grmon3> reg
 * etc...
 */
static inline void trapit(void) {
        __asm__ __volatile__(" ta 1");
}

/*
 * Give external indication on test status.
 *
 * This function is called when a memory verification error is detected. It
 * could be implemented to set a GPIO output used as trig signal for external
 * measurement equipment.
 */
static inline void extsig(unsigned int val)
{
#if 0
        /* Example: Set some GPIO */
        volatile unsigned int *const nim_testvec = (void *) 0x80100510;
        *nim_testvec = val;
#else
        /* By default do nothing */
        (void) val;
#endif
}

static const char *testname[] = {
        "memtest_seqdata",
        "simple write once, read once",
        "simple write once, read forever",
};

static void initstuff();

int main(void)
{
        int col = 0;

        printf("GR716 SRAM example begin\n");
        initstuff();

        extsig(0x00);
        printf(
                "Testing memory in range %08x .. %08x\n",
                (unsigned int) TADDR,
                (unsigned int) TADDR + TSIZE - 1
        );
        printf("testname=%s\n", testname[CFG_TESTNUM]);
        while (1) {
                uint32_t *ret;

                if (CFG_TESTNUM == 0) {
                        ret = memtest_seqdata(TADDR, TSIZE);
                } else if (CFG_TESTNUM == 1) {
                        ret = memtest_simple_a(TADDR, TSIZE, 0);
                } else if (CFG_TESTNUM == 2) {
                        ret = memtest_simple_a(TADDR, TSIZE, 1);
                } else {
                        printf("UNKNOWN TEST\n");
                        return 0;
                }
                if (ret) {
                        extsig(0xff);
                        /* Either trap early or print log as approriate */
                        if (CFG_TRAP_ON_MISMATCH) {
                                trapit();
                        } else {
                                printf("\nFAIL at %08x\n", (unsigned int) ret);
                        }
                } else {
                        printf(".");
                        fflush(NULL);
                }
                col++;
                col &= 0x1f;
                if (!col) {
                        puts("");
                }
        }

        return 0;
}


/*
 * Verify that every memory bit functions. The routine writes size/4 words
 * starting at addr with a pattern, then reads it back and verifies for
 * equality. Another round with inverted pattern is then run. Returns address
 * of the first location which has fails, or NULL if all locations passed.  (A
 * failure at address 0 will also return NULL.)
 *
 * In case the test passes, the memory area is cleared as a side effect.
 */
uint32_t *memtest_seqdata(volatile uint32_t *addr, size_t size)
{
        volatile uint32_t *d;

        d = addr;
        for (uint32_t i = 0; i < size / sizeof(*addr); i++) {
                *d = i;
                d++;
        }

        d = addr;
        for (uint32_t i = 0; i < size / sizeof(*addr); i++) {
                if (*d != i) {
                        return (uint32_t *)d;
                }
                *d = ~i;
                d++;
        }

        d = addr;
        for (uint32_t i = 0; i < size / sizeof(*addr); i++) {
                if (*d != ~i) {
                        return (uint32_t *)d;
                }
                *d = 0;
                d++;
        }

        return NULL;
}

uint32_t *memtest_simple_a(volatile uint32_t *addr, size_t size, int forever)
{
        volatile uint32_t *d;

        /* Fill */
        d = addr;
        for (uint32_t i = 0; i < size / sizeof(*addr); i++) {
                *d = (uint32_t) d;
                d++;
        }

        /* Check */
        do {
                d = addr;
                for (uint32_t i = 0; i < size / sizeof(*addr); i++) {
                        if (*d != (uint32_t) d) {
                                return (uint32_t *)d;
                        }
                        d++;
                }
        } while (forever);

        return NULL;
}


/* Configure the memory controller hosting 2 MiB SRAM */
#include <assert.h>
#include <stdlib.h>
#include <drv/clkgate.h>
#include <drv/gr716/clkgate.h>
#include <bcc/bsp_pnp.h>
#include <gr716/mctrl0_regs.h>
#include <pinhelper.h>

#include <drv/regs/clkgate_bits.h>

static void initstuff(void)
{
        /* clock enable memory controller */
        {
                struct clkgate_priv *clkgate;

                clkgate_init(GR716_CLKGATE_DRV_ALL);
                clkgate = clkgate_open(0);
                assert(clkgate);
                /* reset ftmctrl so we can depend on register reset values */
                clkgate_gate(clkgate, CLKGATE0_GR716_MCTRL0);
                clkgate_enable(clkgate, CLKGATE0_GR716_MCTRL0);
                clkgate_close(clkgate);
        }

        /* configure memory controller for SRAM */
        {
                struct mctrl0_regs *const regs =
                    (struct mctrl0_regs *) GAISLER_FTMCTRL_0_PNP_APB;
                /*
                 * SRAM wait state: 0
                 * SRAM bank size: 8 KiB * 2**8 = 2 MiB
                 */
                regs->mcfg2 = 0x8 << MCTRL0_MCFG2_RAMBANKSZ_BIT;
                /* disable EDAC */
                regs->mcfg3 = 0;
        }

        /* Set pin functions for SRAM */
        {
                /* A0..A18 */
                for (int i = 0; i <= 18; i++) {
                        set_pinfunc(i, IO_MODE_MEM);
                }
                /* A19 */
                set_pinfunc(49, IO_MODE_MEM);
                /* A20 */
                set_pinfunc(50, IO_MODE_MEM);
                /* RAMSN */
                set_pinfunc(19, IO_MODE_MEM);
                /* WRITEN */
                set_pinfunc(34, IO_MODE_MEM);
                /* OEN */
                set_pinfunc(33, IO_MODE_MEM);
                /* D0..D7 */
                for (int i = 25; i <= 32; i++) {
                        set_pinfunc(i, IO_MODE_MEM);
                }
        }
}

