/*    File: bdinit.c
 *  Author: Henrik Gingsjo - Frontgrade Gaisler AB
 * Created: 2023-11-22
 *
 * This is a bdinit file intended for use in GR740 boot images for
 * the GR740-MINI development board with the MKPROM2 bootloader
 * generator. See below links for more information about the products
 * mentioned above:
 *
 *       GR740: https://gaisler.com/gr740
 *  GR740-MINI: https://gaisler.com/gr740-mini
 *     MKPROM2: https://gaisler.com/mkprom2
 *
 * This is needed because the GR740 L2-cache and SDRAM controller are
 * not supported by MKPROM2 itself. To initialize the L2-cache and SDRAM
 * properly before loading an application we extend MKPROM2 through
 * bdinit callbacks. For details refer to the MKPROM2 user manual.
 *
 * This file should be compiled but not linked. I.e pass the -c flag to
 * GCC. Note that the functions will run in a context where neither
 * C runtime nor memory is available. Instead, parameters are passed
 * to functions in registers. Ideally all functions should be leaf
 * procedures (although a call stack up to 8 functions deep can be
 * supported since there are 8 register windows). And they cannot
 * reference the stack or any non-register memory location.
 */

/* In BCC uint32_t is unsigned long. */
#define REGISTER(address) (*((volatile unsigned long*)address))

/* Flash memory controller registers */
/* MCFG2, 4, and 6 are not implemented in the GR740 */
#define REG_MCFG1    REGISTER(0xFF903000)
#define REG_MCFG3    REGISTER(0xFF903008)
#define REG_MCFG5    REGISTER(0xFF903010)
#define REG_MCFG7    REGISTER(0xFF903018)
/* SDRAM controller register */
#define REG_SDCFG1   REGISTER(0xFFE00000)
#define REG_SDCFG2   REGISTER(0xFFE00004)
#define REG_SDMUXCFG REGISTER(0xFFE00020)
/* L2-cache registers */
#define REG_L2CC     REGISTER(0xF0000000)
#define REG_L2CS     REGISTER(0xF0000004)
#define REG_L2CFMA   REGISTER(0xF0000008)
#define REG_L2CACCC  REGISTER(0xF000003C)
/* I/O and PLL configuration registers */
#define REG_FTMEN    REGISTER(0xFFA0B000)
#define REG_ALTEN    REGISTER(0xFFA0B004)
#define REG_LVDSMCLK REGISTER(0xFFA0B008)
/* Clock gating unit (grcg) */
#define REG_UNLOCK   REGISTER(0xFFA04000)
#define REG_ENABLE   REGISTER(0xFFA04004)
#define REG_RESET    REGISTER(0xFFA04008)

/* Timing parameters appropriate for 100 MHz SDRAM. They set the
 * minimum time between certain commands. The one with largest
 * performance implication is TCAS. */
#define SDCFG1_TRP   0
#define SDCFG1_TCAS  0
#define SDCFG1_TRFC  4
/* SDRAM size parameters appropriate for the GR740-MINI
 * It has two banks of 32-bit SDRAM provided by four copies of
 * the IS42S1632F (32Mi x 16). It needs 8192 refresh cycles every 64 ms
 * according to the datasheet. Optionally use twice that rate to be
 * compatible with 50 MHz operation. 100MHz*64ms/8192 = 781.25.
 *
 * There are 32Mi*32/8 = 128 Mi bytes in each bank and the GR740-MINI
 * operates in half-width mode, so BANKSZ=5. The column size is 1024 so
 * COLSZ=2. There are 4 internal banks and 8192 rows so we can reobtain
 * the number of addresses: 4*8192*1024=4*8Mi=32Mi.
 *
 * The COMMAND field _must_ be zero when the controller is enabled.
 * Otherwise an incorrect initialization sequence will be sent.
 */
#define SDCFG1_BANKSZ 5
#define SDCFG1_COLSZ  2
#define SDCFG1_RFLOAD 780

/* SDCFG2 parameters. The GR740-MINI uses double-chip select mode.
 * There are no external pull-resistors so BPARK is recommended (this
 * reduces the power consumption slightly  when the board is idle). */
#define SDCFG2_EN2T  1
#define SDCFG2_DCS   1
#define SDCFG2_BPARK 1

/* 5 instructions in the loop, 250 MHz frequency. If fully pipelined
 * this means we need 50*200 = 10000 loops for 200 us. The SDRAM
 * initialization sequence takes a maximum of 100 SDRAM clock cycles.
 * The L2-cache initialization is worse, taking 5 cycles per cache
 * line. Each way is 0.5 MiB and each line is 32B so there are
 * 1MiB/32B = 32Ki= 2^15 = 0x8000 lines. */
static inline void bdinit_delay(void) {
  __asm__ __volatile__(
    "set	0x8000, %%g1;\n\t"
"1:"
    "nop;\n\t"
    "nop;\n\t"
    "subcc	%%g1,1,%%g1;\n\t"
    "bne	1b;\n\t"
    " nop;\n\t" : : : "g1" );
}

/* configure sdctrl0 */
static inline void bdinit_gr740_sdctrl0()
{
    /* This might be a soft reboot from GRMON. If it is then we should
     * disable the SDRAM controller first. */
    REG_SDCFG1 = 0;
    /* Enable CKE (default after reset), 2T-mode, double chip-select, and bus parking */
    REG_SDCFG2 = (1 << 30) | (SDCFG2_EN2T << 15) | (SDCFG2_DCS << 14) | (SDCFG2_BPARK << 13);
    /* Re-enable SDRAM controller with parameters appropriate for the GR740-MINI */
    REG_SDCFG1 = (1 << 31) | (SDCFG1_TRP << 30) | (SDCFG1_TRFC << 27) | (SDCFG1_TCAS << 26) |
              (SDCFG1_BANKSZ << 23) | (SDCFG1_COLSZ << 21) | /* COMMAND = 0 */
              (SDCFG1_RFLOAD << 0);
    /* NOTE: SDRAM ECC is not supported on the GR740-MINI. But a user
     * might have tried to enable it we explicitly disable it in MUXCFG.
     */
    REG_SDMUXCFG = 0;
    /* After this we should delay until the initialization sequence has
     * completed. In other boards you may see the memory scrubber being
     * used to initialize the SDRAM with known data. But here we are free
     * to skip that because EDAC is disabled. Though clearing the SDRAM
     * could still make sense to wash away a "bad" application that
     * might linger. */
}

static inline void bdinit_gr740_mctrl0()
{
    /* MCFG1 and MCFG3 are handled by MKPROM2. And by GRMON.
     * MCFG5 and MCFG7 need separate treatment. Since BRDYN is not used
     * on the GR740-MINI we only need to bother with MCFG5 which
     * configures lead-out cycles for PROM writes. The IO bus is not
     * on this board either (PCI is used instead). */

     /* Let us use 20 ns. 250 MHz * 20 ns = 5 cycles. */
     REG_MCFG5 = (0 << (7+4)) | (5 << 7);
}


/* enable l2cache again after mkprom disabled it */
static inline void bdinit_gr740_l2cache_invalidate()
{
    /* Disable cache and invalidate all cache lines. After this we must
     * delay until all lines have been invalidate. */
    REG_L2CFMA = (1 << 3) | (5 << 0);
}

static inline void bdinit_gr740_l2cache_on()
{
    /* Default settings except that bypass prefetch is disabled. In
     * particular SPLIT responses are disabled for now. Once workarounds
     * for GRLIB-TN-0021 become implemented in common software we can
     * enable SPLIT again. */
    REG_L2CACCC = (1 << 5);
    /* Enable the cache with default parameters. EDAC disabled. LRU. Not locked. */
    REG_L2CC = (1 << 31) | 0;
}

static inline void bdinit_gr740_grcg_enable(unsigned long coremask)
{
    /* 1. Unlock */
    REG_UNLOCK = coremask;
    /* 2. Assert reset */
    REG_RESET = coremask;
    /* 3. Enable clock */
    REG_ENABLE = coremask;
    /* 4. Disable clock (so that we don't deassert reset in the middle of a clock edge) */
    REG_ENABLE = 0;
    /* 5. Deassert reset */
    REG_RESET = 0;
    /* 6. Reenable clock (safe because static timing) */
    REG_ENABLE = coremask;
    /* 7. Relock the clock gating register */
    REG_UNLOCK = 0;
}

static inline void bdinit_gr740_grcg_disable(unsigned long coremask)
{
    /* 1. Unlock clock gating registers */
    REG_UNLOCK = coremask;
    /* 2. Disable clock. */
    REG_ENABLE = 0;
    /* 3. Relock configuration registers */
    REG_UNLOCK = 0;
    /* Note: If the gated core controls any I/Os they may end up in an
     * undefined state. */
}


static inline void bdinit_gr740_pinsetup()
{
    /* The GR740-MINI has 128MiB of 8-bit PROM provided by a
     * S29GL01GT11DHV023. Thus 27 address lines are used. On reset all
     * are controlled by FTMCTRL. After this function has executed,
     * GPIO2[8:0] will be GPIOs. GPIO2[9] and GPIO2[21]
     * will be UART0. All others remain controlled by FTMCTRL. */

    /* Disable alternate function for GPIO2[8:0] so they can be used as
     * GPIOs. */
    REG_ALTEN = 0x3FFFFF & ~0x1FF;
    /* Disable FTMCTRL function for GPIO2[8:0], GPIO2[9] and GPIO2[21]
     * so they become controlled by GPIO2 and UART0 respectively. */
    REG_FTMEN = 0x3FFFFF & ~((1 << 21) | (1 << 9) | 0x1FF);

   /* NOTE: GPIO2[8:5] control LEDs. GPIO2[4:0] connect to the FPGA.
    * There is no (complete) alternate function mapped to GPIO2[4:0] so
    * there will never be any point to using them as anything other than
    * GPIOs.*/

    /* Now disable unused LVDS pins to save power. Only MEM_CLK_OUT_DIFF
     * is unused. MEM_CLK_OUT (bit 17) is used and all SpaceWires are
     * used. */
    REG_LVDSMCLK = (1 << 17) | (0 << 16) | 0xFF;

    /* SPW4-7 connect to the FPGA. SPW0-3 to the FMC connector. They
     * may be disabled in some applications, but we leave that up to the
     * application software. */
}

/*
 * bdinit0 is called before peripherals have been initialized and
 * before RAM is available.
 */
void bdinit0(void) {
  /* Setup correct lead out cycles for PROM writes */
  bdinit_gr740_mctrl0();
  /* Setup SDRAM controller */
  bdinit_gr740_sdctrl0();
}

/*
 * bdinit1 is called after peripheral registers have been initialized,
 * but before RAM is available.
 */
void bdinit1(void) {
  /* Invalidate L2cache contents and disable cache. */
  bdinit_gr740_l2cache_invalidate();
  /* Wait for cache to be invalidated (and for the SDRAM initialization
   * sequence from bdinit0 to run). */
  bdinit_delay();
  /* Now it is safe to reconfigure and enable L2 cache. */
  bdinit_gr740_l2cache_on();
}

/* bdinit2 is called after MKPROM boot loader has initialized memory. */
void bdinit2(void) {
  /* Setup up PROM/ALT/GPIO pin multiplexing. Disable unused pins. */
  bdinit_gr740_pinsetup();

  /* Enable Clock gated cores */
  /* There is no need to do anything with these because bootstraps
   * already enable/disable everything appropriately. Except maybe
   * L4STAT which is disabled.
   *
   * Enabling a core that is already enabled will cause it to be reset.
   * This may or may not be desireable. Let's avoid resetting the GRETH0
   * SPWRTR and UARTs because a debugger or serial console might be
   * connected through these ports.
   */
  /*bdinit_gr740_grcg_enable(CLKGATE_L4STAT | CLKGATE_SPWRTR); */

  /* UART1 is unused. Disable it.*/
  bdinit_gr740_grcg_disable(1 << 8);
}
