/*    File: helloblink.c
 *  Author: Henrik Gingsjo - Frontgrade Gaisler AB
 * Created: 2023-11-22
 *
 * This is is a small demo application for the GR740-MINI board. It
 * prints to UART0 using regular stdio-functions and blinks four LEDs.
 * It must be compiled with BCC2. Suggested compilation command:
 *
 * sparc-gaisler-elf-gcc -qbsp=gr740 -mcpu=leon3 -O2 helloblink.c -ldrv
 *
 * Explanation of options:
 * -qbsp=gr740: Selects the GR740 BSP. Among other things, this causes
 *              the application to be linked to address 0x00000000 which
 *              is where the RAM is located in the GR740. In older LEON-
 *              systems RAM was traditionally located at 0x40000000.
 * -mcpu=leon3: This option may seem strange since the GR740 uses the
 *              LEON4FT rather than LEON3. But what the option does is
 *              to enable the generation of SPARCV8 mul/div instructions.
 *         -O2: Enable optimizations. Unoptimized SPARC code has huge
 *              overhead since the "keep part of the stack in register
 *              windows" feature of SPARC is practically disabled when
 *              compiling without optimization.
 *       -ldrv: Link against the BCC2 device drivers library (needed for
 *              the GPIO driver functions).
 */

#include <stdio.h>
#include <time.h>
#include <bcc/bcc.h>
#include <drv/gpio.h>

int main()
{
  /* Print on UART0. BCC2 will configure UART0 as the console UART
   * initialization, before it calls main().*/
  printf("Hello from GR740\n");

  /* Install a timer tick interrupt to prevent the time() function from
   * overflowing, and to provide a regular interrupt to wake us up from
   * the low-power mode we will enter below. */
  if(BCC_OK == bcc_timer_tick_init_period(100*1000)) {
    printf("Installed timer tick interrupt at 100 ms intervals\n");
  } else {
    printf("Failed to install timer\n");
  }

  /* Set up GPIO2[8:5] using the GPIO driver included in BCC2. These
   * four GPIOs connect to LEDs on the GR740-MINI. GPIO2[4:0] connect
   * to the FPGA. GPIO[15:0] are unused. GPIO2[21:9] are used by
   * alternate functions (see pin muxing description in the GR740 user
   * manual). */
  gpio_autoinit();
  /* NOTE: The GR740 has two GPIO cores. Core number 0 controls
   * GPIO[15:0]. Core number 1 controls GPIO2[21:0]. */
  printf("%d GPIO cores in the system\n", gpio_dev_count());
  struct gpio_priv *gpio = gpio_open(1);
  if(gpio == NULL) {
    printf("Failed to open device 1\n");
    return 1;
  }
  /* Set the output value of GPIO2[21:0] to 0 (note that all are expected
   * to be inputs at this stage so this has little effect).  */
  gpio_output(gpio, 1, 0);
  /* Set the direction of GPIO2[8:5] to output. All others will be inputs. */
  gpio_direction(gpio, 1, 0xF << 5);

  /* Enter an infinite loop */
  time_t then = time(NULL);
  while(1) {
    /* Print the uptime (in seconds) on UART0 whenever the number of
     * seconds elapsed since boot changes. */
    time_t now = time(NULL);
    if(now != then) {
      printf("uptime: %d\n", (int)now);
      then = now;
    }
    /* Update GPIO outputs. LEDs on the board will reflect the 4 least
     * significant bits of the uptime (measured in seconds). */
    gpio_output(gpio, 1, (now & 0xF) << 5);

    /* Halt in low-power mode. Execution resumes whenever an (unmasked)
     * interrupt occurs. */
    bcc_power_down();
  }

  return 0;
}
