#!/bin/bash
#
# Script: install_32bit_compatibility_package.sh
# 
# This script installs the 32-bit compatibility packages on the 64-bit Linux which
# are required for running Sentinel LDK Runtime Environment on a 64-bit Linux OS.
#
# return codes:
#   0 - Success
#   1 - Missing permissions (must be run as root)
#   2 - Unknown system

echo "Script for installing 32-bit compatibility packages for 64-bit Linux."
echo "Copyright (C) 2017, SafeNet, Inc. All rights reserved."

# Check for Linux
if [ `uname -s` != "Linux" ]; then
    echo "Not running on Linux."
    echo "Aborting..."
    exit 2
fi

# Check for root user
if [ `id -u` -ne 0 ]; then
    echo "This script must run as root."
    echo "Aborting..."
    exit 1
fi

# Check if 32bit support already present
if [ -e /lib/ld-linux.so.2 ]; then
    echo "32-bit support already present. No need to install additional packages."
    exit 0
fi

# Detect hardware platform for x86_64
system_arch=`uname -m`
if [ $system_arch != "x86_64" ]; then
    echo "You can install the x86 32-bit support only on a x86 64-bit platform."
    echo "Aborting..."
    exit 2
fi

# Identifying the package manager
apt-get --help > /dev/null 2>&1
if [ $? -ne 127 ]; then
    package_manager=apt-get
else
    yum --help > /dev/null 2>&1
    if [ $? -ne 127 ]; then
        package_manager=yum
    else
        zypper --help > /dev/null 2>&1
        if [ $? -ne 127 ]; then
            package_manager=zypper
        else
            pacman --help > /dev/null 2>&1
            if [ $? -ne 127 ]; then
                package_manager=pacman
            else
                echo "Unknown package manager."
                echo "Aborting..."
                exit 2
            fi
        fi
    fi
fi

echo -n "Package manager - $package_manager"

# Install the 32 bit support
echo "Installing 32-bit libraries..."
case $package_manager in
yum)
    echo "Executing command: yum install glibc.i686"
    yum install glibc.i686
    ;;
apt-get)
    # Adds a new foreign architecture, removing the need to use the --force option
    # when installing the i386 packages in a x86_64 platform
    # In old Ubuntu, like 12.04, this is going to fail, as multiple architectures were not supported.
    # See: https://wiki.debian.org/Multiarch/Implementation
    echo "Executing command: dpkg --add-architecture i386"
    dpkg --add-architecture i386
    if [ $? -eq 0 ] ; then
        # Install the required libc. The RTE doesn't need anything more to run.
        echo "Executing command: apt-get update ; apt-get install libc6-i386"
        apt-get update
        apt-get install libc6-i386
    else
        # Use the old way to install 32 bits libraries in old not multiarch distributions
        echo "Executing command: apt-get install ia32-libs"
        apt-get install ia32-libs
    fi
    ;;
zypper)
    echo "Executing command: zypper install glibc-32bit"
    zypper install glibc-32bit
    ;;
pacman)
    # Enable multilib support in pacman.conf, to install 32bit libraries on x86_64 Arch Linux platform 
    # Run sync update to sync the packages for multilib support
    # Install required lib32-glibc library on Linux Arch Platform    
    if grep "^\[multilib\]" /etc/pacman.conf; then
        echo "multilib already enabled, running sync update"
    else
	    echo >> /etc/pacman.conf
        echo "[multilib]" >> /etc/pacman.conf
        echo "Include = /etc/pacman.d/mirrorlist" >> /etc/pacman.conf
    fi
    pacman -Syu
    pacman -S lib32-glibc
    ;;
esac

echo "Done"
