/*
 *  AMBA Plug & Play Bus Driver
 *
 *  This driver hook performs bus scanning.
 *
 *  COPYRIGHT (c) 2011.
 *  Aeroflex Gaisler
 *
 *  The license and distribution terms for this file may be
 *  found in the file LICENSE in this distribution or at
 *  http://www.rtems.org/license/LICENSE.
 */

#include <bsp.h>
#include <bsp/fatal.h>
#include <bsp/riscv.h>
#include <grlib/ambapp.h>
#include <grlib/grlib.h>
#include <rtems/sysinit.h>
#include <libcpu/access.h>
#include <amba.h>

uintptr_t grlib_io_area __attribute__((weak)) = 0xfff00000;
unsigned int grlib_timer_prescaler __attribute__((weak)) = 0;
int grlib_timer_core_index __attribute__((weak)) = 0;

/* AMBA Plug&Play information description.
 *
 * After software has scanned AMBA PnP it builds a tree to make
 * it easier for drivers to work with the bus architecture.
 */
struct ambapp_bus ambapp_plb;

/* If RTEMS_DRVMGR_STARTUP is defined extra code is added that
 * registers the GRLIB AMBA PnP bus driver as root driver.
 */
#ifdef RTEMS_DRVMGR_STARTUP
#include <drvmgr/drvmgr.h>
#include <grlib/ambapp_bus_grlib.h>

extern void gptimer_register_drv (void);
extern void apbuart_cons_register_drv(void);
/* All drivers included by BSP, this is overridden by the user by including
 * the drvmgr_confdefs.h. By default the Timer and UART driver are included.
 */
drvmgr_drv_reg_func drvmgr_drivers[] __attribute__((weak)) =
{
  gptimer_register_drv,
  apbuart_cons_register_drv,
  NULL /* End array with NULL */
};

/* Driver resources configuration for AMBA root bus. It is declared weak
 * so that the user may override it, if the defualt settings are not
 * enough.
 */
struct drvmgr_bus_res grlib_drv_resources __attribute__((weak)) =
{
  .next = NULL,
  .resource =
  {
    DRVMGR_RES_EMPTY,
  }
};

/* GRLIB AMBA bus configuration (the LEON3 root bus configuration) */
struct grlib_config grlib_bus_config =
{
  &ambapp_plb,              /* AMBAPP bus setup */
  &grlib_drv_resources,     /* Driver configuration */
};
#endif

uint8_t  _ld8 (uint8_t  *addr) { return *(volatile uint8_t  *) addr; }
uint16_t _ld16(uint16_t *addr) { return *(volatile uint16_t *) addr; }
uint32_t _ld32(uint32_t *addr) { return *(volatile uint32_t *) addr; }
uint64_t _ld64(uint64_t *addr) { return *(volatile uint64_t *) addr; }

void _st8 (uint8_t  *addr, uint8_t  val) { *(volatile uint8_t  *) addr = val; }
void _st16(uint16_t *addr, uint16_t val) { *(volatile uint16_t *) addr = val; }
void _st32(uint32_t *addr, uint32_t val) { *(volatile uint32_t *) addr = val; }
void _st64(uint64_t *addr, uint64_t val) { *(volatile uint64_t *) addr = val; }

/*
 *  amba_initialize
 *
 *  Must be called just before drivers are initialized.
 *  Used to scan system bus. Probes for AHB masters, AHB slaves and
 *  APB slaves. Addresses to configuration areas of the AHB masters,
 *  AHB slaves, APB slaves and APB master are storeds in
 *  amba_ahb_masters, amba_ahb_slaves and amba.
 */

static void amba_initialize(void)
{
  struct ambapp_dev *adev;

  /* Scan AMBA Plug&Play read-only information. The routine builds a PnP
   * tree into ambapp_plb in RAM, after this we never access the PnP
   * information in hardware directly any more.
   * Since on Processor Local Bus (PLB) memory mapping is 1:1
   */
  ambapp_scan(&ambapp_plb, grlib_io_area, NULL, NULL);

  /* find GP Timer */
  adev = (void *) (uintptr_t) ambapp_for_each(&ambapp_plb, (OPTIONS_ALL|OPTIONS_APB_SLVS),
                                 VENDOR_GAISLER, GAISLER_GPTIMER,
                                 ambapp_find_by_idx, &grlib_timer_core_index);
  if (adev) {
    volatile struct gptimer_regs *LEON3_Timer_Regs;
    LEON3_Timer_Regs = (volatile struct gptimer_regs *) (uintptr_t) DEV_TO_APB(adev)->start;

    /* Register AMBA Bus Frequency */
    ambapp_freq_init(
      &ambapp_plb,
      adev,
      (LEON3_Timer_Regs->scaler_reload + 1)
        * 1000 * 1000
    );

    /* Set user prescaler configuration. Use this to increase accuracy of timer
     * and accociated services like cpucounter.
     * Note that minimum value is the number of timer instances present in
     * GRTIMER/GPTIMER hardware. See HW manual.
     */
    if (grlib_timer_prescaler)
      LEON3_Timer_Regs->scaler_reload = grlib_timer_prescaler;
  }

#if 0
  /*
   * NOTE: We are called before bsp_start() so riscv_get_core_freq() is not
   * available yet and will return 0. Otherwise the below could be used instead
   * to register frequency and no GPTIMER is needed. On the other hand, the
   * gptimer method is more general since it decouples the root bus frequency
   * from the CPU.
   */
  {
    unsigned int freq_hz;
    freq_hz = riscv_get_core_frequency();

    /* Register AMBA Bus Frequency */
    if (freq_hz) {
      ambapp_freq_init(&ambapp_plb, NULL, freq_hz);
    }
  }
#endif

#ifdef RTEMS_DRVMGR_STARTUP
  /* Register Root bus, Use GRLIB AMBA PnP bus as root bus */
  ambapp_grlib_root_register(&grlib_bus_config);
#endif
}

RTEMS_SYSINIT_ITEM(
  amba_initialize,
  RTEMS_SYSINIT_BSP_START,
  RTEMS_SYSINIT_ORDER_SECOND
);
